<?php

include_once "elapsed_time.php";

class UDPClient {
  var $host;
  var $port;
  var $sock;
  var $timeout = 5;

  var $response;
  var $seqNum = 0;
  var $reqVer = 2;

  var $debug = true;
  var $errormsg;

  // var $max_packet_size = 65543;
  var $max_retry_times = 10;

  var $type_name_mapping = array(
    'bool'   => 'b',
    'byte'   => 'c',
    'short'  => 's',
    'int'    => 'i',
    'long'   => 'l',
    'float'  => 'f',
    'double' => 'd',
    'Str'    => 'S',
    'Buf'    => 'B'
  );

  function __construct($host, $port)
  {
    $this->host = $host;
    $this->port = $port;
  }

  function calcCrc($buffer)
  {
    $length = strlen($buffer);
    $checksum = 0;
    for($i=0; $i<$length; ++$i)
    {
      $checksum += ord($buffer[$i]);
      $checksum &= 0xffff;
    }
    $checksum ^= 0xffff;
    return chr($checksum/256) . chr($checksum%256);
  }

  function typeShortName($full) 
  {
    if (is_null($full))
      return NULL;

    return isset($this->type_name_mapping[$full]) ? $this->type_name_mapping[$full] : $full;
  }

  function buildHeader(&$buffer, $command)
  {
    $buffer = "CPTW";                 //Protocol ID
    $buffer .= chr($this->reqVer);    //Version

    $this->seqNum = (++$this->seqNum % 256);
    $buffer .= chr($this->seqNum);  //SeqNum

    $buffer .= $command;                   //Command
    return true;
  }

  function buildPayload(&$buffer, $command, $url, $type=NULL, $value=NULL, $short_name=FALSE)
  {
    $length = strlen($url) + 1 + 2;
    $buffer .= chr($length/256);
    $buffer .= chr($length%256);

    if ($command === 'a' or $command === 'w' or $command === 'A' or $command === 'W')
    {
      $buffer .= ($short_name ? "p" : "path") . ":{$url}" . chr(0);

      if ($short_name)
        $buffer .= "t:" . $this->typeShortName($type) . chr(0);
      else
        $buffer .= "type:{$type}" . chr(0);

      //GOTCHA: 'null' in HTTP request's parameter will crash appweb, so use
      //'nil' to represent 'null' value in frontend. here we need to convert it
      //before passing to easyioCpt kit.
      if (($type == "bool" || $type == "float") && $value == "nil")
          $value = "null";
      $buffer .= ($short_name ? "v" : "value") . ":{$value}" . chr(0);
    }
    else
      $buffer .= $url . chr(0);

    return true;
  }
  
  function dataByFullOrShortName($req, $full, $short) 
  {
    if (isset($req[$full]))
      return $req[$full];
    elseif (isset($req[$short]))
      return $req[$short];
    else
      return NULL;
  }

  // the format for multi request: 
  //  multiple write: 
  //    W + reqNum(1 byte) + reqList
  //    reqList: path:PATH1\0type:TYPE1\0value:VALUE1\0...
  //  multiple action: 
  //    A + reqNum(1 byte) + reqList
  //    reqList: path:PATH1\0type:TYPE1\0value:VALUE1\0...
  //
  // to save memory space in multi request, we will use short names
  // for both keys and some value: 
  // keys: 'p' => 'path', 't' => 'type', 'v' => 'value'
  // values: possible type values: 
  //   'b' => bool, 'c' => 'byte', 's' => short, 'i' => int, 'l' => long
  //   'f' => float, 'd' => 'double', 'B' => Buf, 'S' => Str
  function buildMultiRequest($command, $req_list) 
  {
    $buffer = "";
    $this->buildHeader($buffer, $command);

    $req_count = count($req_list);
    if ($req_count >= 255)
      return NULL;

    $buffer .= chr($req_count);
    foreach($req_list as $req) {
      $this->buildPayload($buffer, $command, 
        $this->dataByFullOrShortName($req, 'path', 'p'),
        $this->dataByFullOrShortName($req, 'type', 't'),
        $this->dataByFullOrShortName($req, 'value', 'v'),
      TRUE);
    }
    $buffer .= $this->calcCrc($buffer);
    return $buffer;
  }

  function buildRequest($command, $url, $type=NULL, $value=NULL)
  {
    $buffer = "";
    $this->buildHeader($buffer, $command);
    $this->buildPayload($buffer, $command, $url, $type, $value);
    $buffer .= $this->calcCrc($buffer);
    return $buffer;
  }
  
  function downgradeRequest(&$request) 
  {
    //the request's minimum size should be 12
    if (strlen($request) < 12)
      return false;
    
    $curReqVer = ord($request[4]);
    if ($curReqVer != $this->reqVer)
      return false;

    $request[4] = chr($curReqVer - 1);

    $checksum = $this->calcCrc(substr($request, 0, -2));
    $request = substr_replace($request, $checksum, -2, 2);
    
    return true;
  }

  function send($request)
  {
    $startTime = microTime(true);

    if (!$fp = fsockopen("udp://{$this->host}", $this->port, $errno, $errstr, $this->timeout))
    {
      // Set error message
      switch($errno) {
      case -3:
        $this->errormsg = 'Socket creation failed (-3)';
      case -4:
        $this->errormsg = 'DNS lookup failure (-4)';
      case -5:
        $this->errormsg = 'Connection refused or timed out (-5)';
      default:
        $this->errormsg = 'Connection failed ('.$errno.')';
      }
      $this->errormsg .= ' '.$errstr;
      $this->debug($this->errormsg);
      return false;
    }

    stream_set_timeout($fp, $this->timeout);
    $result = fwrite($fp, $request);

    // error_log("### send request: " . substr($request, 9, -2));

    $needDowngrade = $this->fetchResponse($fp) == -1;
    if ($needDowngrade && $this->downgradeRequest($request))
    {
      stream_set_timeout($fp, $this->timeout);
      $result = fwrite($fp, $request);
      $this->fetchResponse($fp);
    }

    fclose($fp);

    //### for testing
    // $this->dumpPacket($this->response);
   
    $endTime = microTime(true);
    //$logMsg = sprintf('%.4f seconds, %.2f kb', ($endTime - $startTime), strlen($this->response)/1024.0);
    //error_log('UDP request: ' . $logMsg);

    return true;
  }
  
  //return 0 means everything is good
  //return -1 means the protocol mismatch, need to downgrade and send request again
  function fetchResponse($fp) 
  {
    // Reset all the variables that should not persist between requests
    $this->response = '';
    $this->errormsg = '';

    // $this->response = fread($fp, 65545);

    $retryTimes = $this->max_retry_times;

    //NOTE: app.scanPeriod value decides how often the svm loop cycle runs. for
    //example, set scanPeriod to 200, then svm cycle will run once at most
    //every 200ms. if a cycle runs for 50ms, then svm will wait for
    //150ms(200-50) before the next cycle. because of this, if a request arrive
    //just after a cycle ends, it will need to wait for 200ms at most before
    //next cycle. the request handling is pretty fast in svm, for single object
    //Add2, it takes ~10ms to return the result. but the cyle may make it much slower 
    stream_set_timeout($fp, 1, 0); // use shorter timeout value
    // while (!feof($fp)) { //when server doesn't close socket, this will always be False
    while (true) {
      $content = fread($fp, 8192);
      if ($content == false)
      {
        if ($retryTimes <= 0)
        {
            error_log("request timed out {$this->max_retry_times} times, giving up...");
            break;
        }
        else
        {
            $retryTimes -= 1;
            continue;
        }
      }

      if (substr($content, -2) === "\0\0")
      {
          $this->response .= substr($content, 0, -2);
          break;
      }
      else
          $this->response .= $content;
    }

    $respLen = strlen($this->response);
    if ( $respLen > 90 && $respLen < 150 && strpos($this->response, '"errorCode": 99') )
        return -1;
    else
        return 0;
  }

  function get($url)
  {
    $request = $this->buildRequest('r', $url);
    return $this->send($request);
  }

  function invokeAction($url, $type, $value)
  {
    $request = $this->buildRequest('a', $url, $type, $value);
    return $this->send($request);
  }

  function writeProperty($url, $type, $value)
  {
    $request = $this->buildRequest('w', $url, $type, $value);
    return $this->send($request);
  }

  function doMultiRequest($is_action, $req_list) 
  {
    $request = $this->buildMultiRequest($is_action ? 'A' : 'W', $req_list);
    return $this->send($request);
  }

  function getContent()
  {
    if (strlen($this->response) <= 11)
    {
      if (strlen($this->response) == 0)
        $this->debug("response is empty");
      else
        $this->debug("response's size is too small" . $this->response);
      return $this->genErrorResponse("invalid response size");
    }

    $protocolID = substr($this->response, 0, 4);
    if ($protocolID != "CPTW")
    {
      $this->debug("invalid protocol ID: " . $protocolID);
      return $this->genErrorResponse("invalid response protocol");
    }

    $version = ord(substr($this->response, 4, 1));
    $seqNum = ord(substr($this->response, 5, 1));
    if ($seqNum != $this->seqNum)
    {
      $this->debug("returned seqNum({$seqNum}) does not match original seqNum({$this->seqNum})");
      return $this->genErrorResponse("invalid response seq number");
    }

    $command = substr($this->response, 6, 1);

    if ($version == 1)
      return $this->handleVersion1();
    else if ($version == 2)
      return $this->handleVersion2();
    else if ($version == 3)
      return $this->handleVersion3(); 
    else
    {
        $this->debug("invalid version number: " . $version);
        return $this->genErrorResponse("invalid response version");
    }
  }

  function handleVersion1()
  {
    $length = ord(substr($this->response, 7, 1))*256 + ord(substr($this->response, 8, 1));
    $content = substr($this->response, 9, -2);
    if (strlen($content) != $length-2)
    {
      $this->debug("expected content's length is {${$length-2}}, but it is {${strlen($content)}}");
      return $this->genErrorResponse("invalid response length(v1)");
    }

    if ($this->calcCrc(substr($this->response, 0, -2)) != substr($this->response, -2))
    {
      $this->debug("invalid crc checksum");
      return $this->genErrorResponse("invalid response checksum(v1)");
    }

    return substr($content, 0, -1); //get rid of ending '\0'
  }

  function handleVersion2()
  {
    $content = substr($this->response, 7, -4);
    $length = ord(substr($this->response, -4, 1))*256 + ord(substr($this->response, -3, 1));
    if (strlen($content) != $length-4)
    {
      $this->debug("expected content's length is " . ($length-4) . ", but it is " . strlen($content));
      return $this->genErrorResponse("invalid response length(v2)");
    }

    $crc = substr($this->response, -2);
    if ($this->calcCrc(substr($this->response, 0, -2)) != $crc)
    {
      //NOTE: need to check if there are too many packet loss, if yes, need to
      //improve the protocal with Koh, maybe easyioCpt kit sends a packet,
      //then php client ask more, then send one more packet, on and on.
      //untill the last packet received.
      $this->debug("invalid crc checksum, expected: " . $crc);
      return $this->genErrorResponse("invalid response checksum(v2)");
    }

    return substr($content, 0, -1); //get rid of ending '\0'
  }
  
  function handleVersion3()
  {
    $content = substr($this->response, 7, -6);

    //for version3, 'length' field is 4bytes 
    $length = ord(substr($this->response, -6, 1));
    $length = $length*256 + ord(substr($this->response, -5, 1));
    $length = $length*256 + ord(substr($this->response, -4, 1));
    $length = $length*256 + ord(substr($this->response, -3, 1));

    if (strlen($content) != $length-6)
    {
      $this->debug("expected content's length is " . ($length-6) . ", but it is " . strlen($content));
      return $this->genErrorResponse("invalid response length(v3)");
    }

    $crc = substr($this->response, -2);
    if ($this->calcCrc(substr($this->response, 0, -2)) != $crc)
    {
      //NOTE: need to check if there are too many packet loss, if yes, need to
      //improve the protocal with Koh, maybe easyioCpt kit sends a packet,
      //then php client ask more, then send one more packet, on and on.
      //untill the last packet received.
      $this->debug("invalid crc checksum, expected: " . $crc);
      return $this->genErrorResponse("invalid response checksum(v3)");
    }

    return substr($content, 0, -1); //get rid of ending '\0'
  }
  
  function genErrorResponse($errmsg) { 
    $resp = array('resultCode' => 1, 'errorMsg' => $errmsg);
    return $this->map2json($resp);
  }

  function getError()
  {
    return $this->errormsg;
  }

  function setDebug($debug)
  {
    $this->debug = $debug;
  }

  function debug($msg, $object = false) {
    error_log($msg);

    // if ($this->debug) {
    //   print '<div style="border: 1px solid red; padding: 0.5em; margin: 0.5em;"><strong>UDPClient Debug:</strong> '.$msg;
    //   if ($object) {
    //     ob_start();
    //     print_r($object);
    //     $content = htmlentities(ob_get_contents());
    //     ob_end_clean();
    //     print '<pre>'.$content.'</pre>';
    //   }
    //   print '</div>';
    // }
  }

  function dumpPacket($response) {
    $handle = fopen("packet.log", "a+");
    fwrite($handle, $response);
    fwrite($handle, "\n");
    fclose($handle);
  }

  //following functions will handle json encoding stuff for backward
  //compatible, just borrowed from utils.php file
  protected function isAssoc($array) {
    $keys = array_keys($array);
    return array_keys($keys) !== $keys;
  }

  protected function list2json($listData) {
    $json = array();
    foreach($listData as $item) {
        if (is_string($item))
          $item = addcslashes($item,"\\\"\n\r");
        else if (is_array($item)) {
          if ($this->isAssoc($item))
            $item = $this->map2json($item);
          else
            $item = $this->list2json($item);
          $json[] = $item;
          continue;
        }
        $json[] = '"' . $item . '"';
    }
    return '[' . implode(',', $json) . ']';
  }

  protected function map2json($mapData) {
    if (function_exists("json_encode"))
      return json_encode($mapData);

    $json = array();
    foreach($mapData as $k => $v) {
      if (is_string($k))
        $k = addcslashes($k,"\\\"\n\r");

      if (is_string($v))
        $v = addcslashes($v,"\\\"\n\r");
      else if (is_bool($v)) {
        $v = $v ? 'true' : 'false';
        $json[] = "\"$k\": $v";
        continue;
      }
      else if (is_array($v)) {
        if ($this->isAssoc($v))
          $v = $this->map2json($v);
        else
          $v = $this->list2json($v);

        $json[] = "\"$k\": $v";
        continue;
      }
      $json[] = "\"$k\": \"$v\"";
    }
    return '{' . implode(',', $json) . '}';
  }

}

?>
