$(function() {

	var netmaskList = [ //{{{1
		"0.0.0.0", "128.0.0.0", "192.0.0.0", "224.0.0.0",
		"240.0.0.0", "248.0.0.0", "252.0.0.0", "254.0.0.0",
		"255.0.0.0", "255.128.0.0", "255.192.0.0", "255.224.0.0",
		"255.240.0.0", "255.248.0.0", "255.252.0.0", "255.254.0.0",
		"255.255.0.0", "255.255.128.0", "255.255.192.0", "255.255.224.0",
		"255.255.240.0", "255.255.248.0", "255.255.252.0", "255.255.254.0",
		"255.255.255.0", "255.255.255.128", "255.255.255.192", "255.255.255.224",
		"255.255.255.240", "255.255.255.248", "255.255.255.252", "255.255.255.254",
	]; //}}}1

  //define components 
  var generalComp = { //{{{1
    template: '#general-network-template',
    data: function() {
      return {
        hostname: '',

        hostnameError: false,
        ipError: false,
        netmaskError: false,
        gatewayError: false,
        dns1Error: false,
        dns2Error: false,

        currentView: 'manual',
        sections: [
          {id: 'manual', name: 'Manually'},
          {id: 'auto', name: 'Using DHCP'},
        ],

        spinner: null,
        manualSettings: {
          ip: '',
          netmask: '255.255.255.0',
          gateway: '',
          dns1: '',
          dns2: '',
        },
        autoSettings: {
          ip: '',
          netmask: '',
          gateway: '',
          dns1: '',
          dns2: '',
        },
      };
    },
    mounted: function() {
      var self = this;
      $.ajax({
        url: 'network_config.php',
        type: 'GET', 
        dataType: 'json',
        data: { type: 'general' },
        beforeSend: function() {
          self.getSpinner().spin(self.$el);
        },
        success: function(resp) {
          if (resp.redirect)
            redirect(resp.redirect);

          if (resp.data) {
            if (_.has(resp.data, 'hostname'))
              self.hostname = resp.data['hostname'];
            var addresses = resp.data['ipv4.addresses'];
            var parts = addresses.split("/");
            if (parts.length == 2) {
              self.manualSettings.ip = parts[0];
              var index = Number(parts[1]);
              if (index >= 0 && index < netmaskList.length)
                self.manualSettings.netmask = netmaskList[index];
            }
            self.manualSettings.gateway = resp.data['ipv4.gateway'];

            var ipv4DNS = resp.data['ipv4.dns'];
            if (ipv4DNS) {
              var dnsList = _.map(ipv4DNS.split(','), function(part) { return part.trim() });
              if (dnsList.length > 0 && self.validateIP(dnsList[0]))
                self.manualSettings.dns1 = dnsList[0];
              if (dnsList.length > 1 && self.validateIP(dnsList[1]))
                self.manualSettings.dns2 = dnsList[1];
            }

            var method = resp.data['ipv4.method'];
            if (method == 'auto') {
              var dhcpOptions = _.reduce(_.keys(resp.data), function(memo, key) {
                if (!key.startsWith("DHCP4.OPTION"))
                  return memo;

                var parts = resp.data[key].trim().split("=");
                if (parts.length != 2)
                  return memo;
                memo[parts[0].trim()] = parts[1].trim();
                return memo;
              }, {});

              self.autoSettings.ip = dhcpOptions['ip_address'];
              self.autoSettings.gateway = _.first(dhcpOptions['routers'].split(','));
              self.autoSettings.netmask = dhcpOptions['subnet_mask'];
              self.autoSettings.dns1 = _.first(dhcpOptions['domain_name_servers'].split(','));
            }

            self.method = method;
          }
        },
        error: function(data, status) {
          console.warn("get error");
          console.warn(data);
        },
        complete: function() {
          self.getSpinner().stop();
        }
      });
    },
    computed: {
      ip: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.ip : this.autoSettings.ip;
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.ip = v;
        },
      },
      netmask: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.netmask : this.autoSettings.netmask;
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.netmask = v;
        },
      },
      gateway: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.gateway : this.autoSettings.gateway;
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.gateway = v;
        },
      },
      dns1: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.dns1 : this.autoSettings.dns1;
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.dns1 = v;
        },
      },
      dns2: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.dns2 : this.autoSettings.dns2;
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.dns2 = v;
        },
      },

      cdir: function() {
        return netmaskList.indexOf(this.netmask);
      },
      method: {
        get: function() {
          return this.currentView;
        },
        set: function(v) {
          if (v == 'manual')
            this.currentView = 'manual';
          else if (v == 'auto')
            this.currentView = 'auto';
        },
      },
    },
    methods: {
      getSpinner: function() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },

      validateIP: function(ipaddr) {
        if (!ipaddr)
          return false;

        if (ipaddr == "0.0.0.0" || ipaddr == "255.255.255.255")
          return false;

        var pat = /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/;
        var parts = ipaddr.match(pat);
        if (parts == null || parts.length != 5)
          return false;

        parts.shift();
        if (_.any(parts, function(p) { return parseInt(p) > 255; }))
          return false;

        return true;
      },

      validateMask: function(netmask) {
        if (!this.validateIP(netmask))
          return false;
				return _.contains(netmaskList, netmask);
      },

      validateInputs: function() {
        this.hostnameError = !this.hostname && !/^[-a-zA-Z0-9]{1,253}$/.test(this.hostname);

        if (this.method == 'auto')
          return !this.hostnameError;

        this.ipError = !this.validateIP(this.ip);
        this.netmaskError = !this.validateMask(this.netmask);
        this.gatewayError = !this.validateIP(this.gateway);
        if (this.dns1.trim().length > 0)
          this.dns1Error = !this.validateIP(this.dns1);
        else
          this.dns1Error = false;
        if (this.dns2.trim().length > 0)
          this.dns2Error = !this.validateIP(this.dns2);
        else
          this.dns2Error = false;

        return !(this.hostnameError || this.ipError || this.netmaskError || this.gatewayError || this.dns1Error || this.dns2Error);
      },

      saveGeneralNetworkSettings: function() {
        if (!this.validateInputs()) {
          console.warn("invalid input");
          return;
        }

        var self = this;
        var ipv4_data = {
          method: this.method,
        };
        if (this.method == 'manual') {
          var ipv4DNS = _.map(_.reject([this.dns1, this.dns2], function(dns) { return !self.validateIP(dns); }), 
            function(dns) { return dns.trim(); }).join(',');
          _.extend(ipv4_data, {
            ip: this.ip,
            cdir: this.cdir,
            gateway: this.gateway,
            dns: ipv4DNS
          });
        }
        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data: {
            type: 'general',
            config: {
              //GOTCHA: now the firmware will always set the default hostname after
              //reboot, when the issue has been fixed later, we can enable modifyHostName 
              // hostname: this.hostname,
              ipv4: ipv4_data
            },
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data) {
            if (data.redirect)
              redirect(data.redirect);

            var msg = "settings have been saved successfully.";
            if (self.method == 'manual')
              msg = msg + " redirecting ...";
            msgbus.$emit("alert", "Done", msg, 'success'); 
            if (self.method == 'manual')
              _.delay(function() {
                var l = window.location;
                var url = l.href;
                window.location.assign(url.replace(l.hostname, self.ip));
              }, 2500);
          },
          error: function(data, status) {
            msgbus.$emit("alert", "Error", "failed to save settings.", 'error'); 
          },
          complete: function(data, status) {
            self.getSpinner().stop();
          },
        });
      }
    },
  }; //}}}1

  var progressUIMixin = { //{{{1
    data: function() {
      return {spinner: null};
    },
    methods: {
      getSpinner: function() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },
    },
  }; //}}}1

  var fileSelectorInputMixin = { //{{{1
    methods: {
      changeFile: function(title, prop) {
        var inst = this.$root.$refs['file_selector_inst'];
        var oldVal = this.$data[prop];
        var self = this;
        inst.doModal(title, oldVal, function(newVal) {
          self.$set(self.$data, prop, newVal);
        });
      },
    },
  }; //}}}1

  var md5Comp = { //{{{1
    template: '#security-802_1x-md5-template',
    props: ['enabled', 'initialSettings'],
    data: function() {
      return {
        username: '',
        password: '',

        usernameError: false,
        passwordError: false,
      };
    },
    mounted: function() {
      this.configData = this.initialSettings;
    },
    computed: {
      configData: {
        get: function() {
          return {
            identity: this.username, 
            password: this.password
          };
        },
        set: function(newVal) {
          if (_.has(newVal, '802-1x.identity'))
            this.username = newVal['802-1x.identity'];
          else
            this.username = '';

          if (_.has(newVal, '802-1x.password'))
            this.password = newVal['802-1x.password'];
          else
            this.password = '';
        },
      },
    },
    methods: {
      validate: function() {
        this.usernameError = this.username.length <= 0;
        //TODO: check if password is secure enough
        this.passwordError = this.password.length < 8;
        return !this.usernameError && !this.passwordError;
      },
    },
  }; //}}}1

  var tlsComp = { //{{{1
    mixins: [fileSelectorInputMixin],
    template: '#security-802_1x-tls-template', 
    props: ['enabled', 'initialSettings'],
    data: function() {
      return {
        identity: '',
        cert: '',
        caCert: '',
        key: '',
        keyPassword: '',
        
        identityError: false,
        certError: false,
        caCertError: false,
        keyError: false,
        keyPasswordError: false,
      };
    },
    mounted: function() {
      this.configData = this.initialSettings;
    },
    computed: {
      configData: {
        get: function() {
          return {
            'identity': this.identity, 
            'client-cert': this.cert, 
            'ca-cert': this.caCert,
            'private-key': this.key,
            'private-key-password': this.keyPassword,
          };
        },
        set: function(newVal) {
          if (_.has(newVal, '802-1x.identity'))
            this.identity = newVal['802-1x.identity'];
          else
            this.identity = '';

          if (_.has(newVal, '802-1x.client-cert'))
            this.cert = newVal['802-1x.client-cert'];
          else
            this.cert = '';

          if (_.has(newVal, '802-1x.ca-cert'))
            this.caCert = newVal['802-1x.ca-cert'];
          else
            this.caCert = '';

          if (_.has(newVal, '802-1x.private-key'))
            this.key = newVal['802-1x.private-key'];
          else
            this.key = '';

          if (_.has(newVal, '802-1x.private-key-password'))
            this.keyPassword = newVal['802-1x.private-key-password'];
          else
            this.keyPassword = '';
        },
      },
    },
    methods: {
      validate: function() {
        this.identityError = this.identity.length <= 0;
        this.certError = this.cert.length <= 0;
        // this.caCertError = this.caCert.length <= 0;
        this.keyError = this.key.length <= 0;
        this.keyPasswordError = this.keyPassword.length <=8;

        return _.any([this.identityError, this.certError, /*this.caCertError,*/ this.keyError, this.keyPassword]);
      },
    },
  }; //}}}1

  var ttlsComp = { //{{{1
    mixins: [fileSelectorInputMixin],
    template: '#security-802_1x-ttls-template', 
    props: ['enabled', 'initialSettings'],
    data: function() {
      return {
        anonymousIdentity: '',
        anonymousIdentityError: false,

        caCert: '',
        caCertError: false,
        
        identity: '',
        identityError: false,

        password: '',
        passwordError: false,

        phase2AuthEap: 'pap',
        phase2AuthEapList: ['PAP', 'MSCHAP', 'MSCHAPv2', 'CHAP', 'MD5', 'GTC'],
      };
    },
    mounted: function() {
      this.configData = this.initialSettings;
    },
    computed: {
      configData: {
        get: function() {
          let conf = {
            'anonymous-identity': this.anonymousIdentity, 
            'ca-cert': this.caCert,
            'identity': this.identity,
            'password': this.password,
          };

          if (_.contains(['pap', 'mschap', 'chap'], this.phase2AuthEap))
            conf['phase2-auth'] = this.phase2AuthEap;
          else
            conf['phase2-autheap'] = this.phase2AuthEap;
          return conf;
        },
        set: function(newVal) {
          if (_.has(newVal, '802-1x.identity'))
            this.identity = newVal['802-1x.identity'];
          else
            this.identity = '';

          if (_.has(newVal, '802-1x.password'))
            this.password = newVal['802-1x.password'];
          else
            this.password = '';

          if (_.has(newVal, '802-1x.ca-cert'))
            this.caCert = newVal['802-1x.ca-cert'];
          else
            this.caCert = '';

          if (_.has(newVal, '802-1x.anonymous-identity'))
            this.anonymousIdentity = newVal['802-1x.anonymous-identity'];
          else
            this.anonymousIdentity = '';
        
          var phase2Auth = '';
          if (_.has(newVal, '802-1x.phase2-autheap'))
            phase2Auth = newVal['802-1x.phase2-autheap'];
          if (phase2Auth.length == 0 && _.has(newVal, '802-1x.phase2-auth'))
            phase2Auth = newVal['802-1x.phase2-auth'];

          if (phase2Auth.length == 0)
            phase2Auth = 'pap';
          this.phase2AuthEap = phase2Auth;
        },
      },
    },
    methods: {
      validate: function() {
        this.anonymousIdentityError = this.anonymousIdentity.length <= 0;
        this.caCertError = this.caCert.length <= 0;
        this.identityError = this.identity.length <= 0;
        this.passwordError = this.password.length <= 8;

        return _.any([this.identityError, this.caCertError, this.passwordError, this.anonymousIdentityError]);
      },
    },
  }; //}}}1

  var peapComp = { //{{{1
    mixins: [fileSelectorInputMixin],
    template: '#security-802_1x-peap-template', 
    props: ['enabled', 'initialSettings'],
    data: function() {
      return {
        anonymousIdentity: '',
        anonymousIdentityError: false,

        caCert: '',
        caCertError: false,
        
        identity: '',
        identityError: false,

        password: '',
        passwordError: false,

        phase2Auth: 'mschapv2',
        phase2AuthList: ['MSCHAPv2', 'MD5', 'GTC'],
      };
    },
    mounted: function() {
      this.configData = this.initialSettings;
    },
    computed: {
      configData: {
        get: function() {
          return {
            'anonymous-identity': this.anonymousIdentity, 
            'ca-cert': this.caCert,
            'identity': this.identity,
            'password': this.password,
            'phase2-auth': this.phase2Auth,
          };
        },
        set: function(newVal) {
          if (_.has(newVal, '802-1x.identity'))
            this.identity = newVal['802-1x.identity'];
          else
            this.identity = '';

          if (_.has(newVal, '802-1x.password'))
            this.password = newVal['802-1x.password'];
          else
            this.password = '';

          if (_.has(newVal, '802-1x.ca-cert'))
            this.caCert = newVal['802-1x.ca-cert'];
          else
            this.caCert = '';

          if (_.has(newVal, '802-1x.anonymous-identity'))
            this.anonymousIdentity = newVal['802-1x.anonymous-identity'];
          else
            this.anonymousIdentity = '';
        
          if (_.has(newVal, '802-1x.phase2-auth'))
            this.phase2Auth = newVal['802-1x.phase2-auth'];
          else
            this.phase2Auth = 'mschapv2';
        },
      },
    },
    methods: {
      validate: function() {
        this.anonymousIdentityError = this.anonymousIdentity.length <= 0;
        this.caCertError = this.caCert.length <= 0;
        this.identityError = this.identity.length <= 0;
        this.passwordError = this.password.length <= 8;

        return _.any([this.identityError, this.caCertError, this.passwordError, this.anonymousIdentityError]);
      },
    },
  }; //}}}1

  var securityComp = { //{{{1
    template: '#security-802_1x-template',
    mixins: [progressUIMixin],
    data: function() {
      return {
        enabled: false,
        currentView: 'md5',
        sections: [
          {id: 'md5', name: 'MD5'},
          {id: 'tls', name: 'TLS'},
          {id: 'ttls', name: 'Tunneled TLS'},
          {id: 'peap', name: 'Protected EAP(PEAP)'}
        ],
        initialSettings: {},
      };
    },
    mounted: function() {
      this.loadSecuritySettings();
    },
    methods: {
      loadSecuritySettings: function() {
        //TODO: load security settings from server
        var self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'GET', 
          dataType: 'json',
          data: { type: '802.1x' },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (resp.data) {
              self.initialSettings = resp.data;
              if (_.has(self.initialSettings, '802-1x.eap')) {
                if (self.initialSettings['802-1x.eap'].length > 0) {
                  self.currentView = self.initialSettings['802-1x.eap'];
                  self.enabled = true;
                }
                else
                  self.enabled = false;
              }
              self.$refs['sub_config_panel'].configData = self.initialSettings;
            }
            else
              self.enabled = false;
          },
          error: function(data, status) {
            console.warn(data);
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },
      saveSecuritySettings: function() {
        var panel = this.$refs['sub_config_panel'];
        if (!panel) {
          console.warn("can not find current panel");
          return;
        }
          
        // if (this.enabled && !panel.validate())
        //   return

        var data = { type: '802.1x' };
        if (this.enabled)
          _.extend(data, {
            eap: this.currentView,
            config: panel.configData,
          });
        else
          _.extend(data, { eap: '' });

        var self = this;
        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data: data,
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data) {
            if (data.redirect)
              redirect(data.redirect);

            var msg = "security settings have been saved successfully.";
            msgbus.$emit("alert", "Done", msg, 'success'); 

            if (window.iot_platform == 'FW') 
              _.delay(function() { window.history.back(); }, 1500);
          },
          error: function(data, status) {
            msgbus.$emit("alert", "Error", "failed to save security settings.", 'error'); 
          },
          complete: function(data, status) {
            self.getSpinner().stop();
          },
        });
      },
    },
    components: {
      'md5':  md5Comp,
      'tls':  tlsComp,
      'ttls': ttlsComp,
      'peap': peapComp,
    },
  }; //}}}1

  var ipv6Comp = { //{{{1
    template: '#ipv6-network-template',
    data: function() {
      return {
        ipError: false,
        subnetPrefixLenError: false,
        gatewayError: false,
        dns1Error: false,
        dns2Error: false,

        manualSettings: {
          ip: '',
          subnetPrefixLen: 64,
          gateway: '',
          dns1: '',
          dns2: ''
        },
        autoSettings: {
          ipList: [],
          gateway: '',
          dnsList: [],
        },

        currentView: 'manual',
        sections: [
          {id: 'manual', name: 'Manual'},
          {id: 'auto', name: 'Automatic'},
        ],

        spinner: null,
      };
    },
    mounted: function() {
      var self = this;
      $.ajax({
        url: 'network_config.php',
        type: 'GET', 
        dataType: 'json',
        data: { type: 'ipv6' },
        beforeSend: function() {
          self.getSpinner().spin(self.$el);
        },
        success: function(resp) {
          if (resp.redirect)
            redirect(resp.redirect);

          if (resp.data) {
            var addresses = _.compact(resp.data['ipv6.addresses'].split(','));
            if (addresses.length >= 1) {
              var parts = addresses[0].split("/");
              if (parts.length == 2) {
                if (self.validateIP(parts[0]))
                  self.manualSettings.ip = parts[0];

                var prefixLen = Number(parts[1]);
                if (prefixLen >= 1 && prefixLen <= 128)
                  self.manualSettings.subnetPrefixLen = prefixLen;
              }
            }
            if (_.has(resp.data, 'ipv6.gateway'))
              self.manualSettings.gateway = resp.data['ipv6.gateway'];

            var ipv6DNS = resp.data['ipv6.dns'];
            if (ipv6DNS) {
              var dnsList = _.map(ipv6DNS.split(','), function(part) { return part.trim() });
              if (dnsList.length > 0 && self.validateIP(dnsList[0]))
                self.manualSettings.dns1 = dnsList[0];
              if (dnsList.length > 1 && self.validateIP(dnsList[1]))
                self.manualSettings.dns2 = dnsList[1];
            }

            var method = resp.data['ipv6.method'];
            if (method == 'auto') {
              self.autoSettings.gateway = resp.data['IP6.GATEWAY'];
              _.each(
                _.select(_.keys(resp.data), function(k) { return k.startsWith("IP6.ADDRESS"); }),
                function(elem, index) {
                  self.autoSettings.ipList.push(resp.data[elem]);
                });
              _.each(
                _.select(_.keys(resp.data), function(k) { return k.startsWith("IP6.DNS"); }),
                function(elem, index) {
                  self.autoSettings.dnsList.push(resp.data[elem]);
                });
            }

            self.method = method;
          }
        },
        error: function(data, status) {
          console.warn("get error");
          console.warn(data);
        },
        complete: function() {
          self.getSpinner().stop();
        }
      });
    },
    computed: {
      ip: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.ip : '';
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.ip = v;
        },
      },
      subnetPrefixLen: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.subnetPrefixLen : '';
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.subnetPrefixLen = v;
        },
      },
      gateway: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.gateway : '';
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.gateway = v;
        },
      },
      dns1: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.dns1 : '';
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.dns1 = v;
        },
      },
      dns2: {
        get: function() {
          return this.currentView == 'manual' ? this.manualSettings.dns2 : '';
        },
        set: function(v) {
          if (this.currentView == 'manual')
            this.manualSettings.dns2 = v;
        },
      },

      method: {
        get: function() {
          return this.currentView;
        },
        set: function(v) {
          if (v == 'manual')
            this.currentView = 'manual';
          else if (v == 'auto')
            this.currentView = 'auto';
        },
      },
    },
    methods: {
      getSpinner: function() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },

      validateIP: function(ipaddr) {
        if (!ipaddr)
          return false;

        var pat = /^[0-9a-f]{0,4}$/;
        var parts = ipaddr.split(":");
        if (parts.length < 3 || parts.length > 8)
          return false;

        return _.all(parts, function(part) {
          return pat.test(part);
        })
      },
      validateInputs: function() {
        this.ipError = !this.validateIP(this.ip);
        this.subnetPrefixLenError = this.subnetPrefixLen > 128 || this.subnetPrefixLen < 1;
        this.gatewayError = !this.validateIP(this.gateway);
        if (this.dns1.trim().length > 0)
          this.dns1Error = !this.validateIP(this.dns1);
        else
          this.dns1Error = false;
        if (this.dns2.trim().length > 0)
          this.dns2Error = !this.validateIP(this.dns2);
        else
          this.dns2Error = false;

        return !(this.ipError || this.subnetPrefixLenError || this.gatewayError || this.dns1Error || this.dns2Error);
      },

      saveIPv6NetworkSettings: function() {
        if (this.method == 'manual' && !this.validateInputs()) {
          console.warn("invalid input");
          return;
        }

        var self = this;
        var ipv6_data = {
          method: this.method,
        };
        if (this.method == 'manual') {
          var ipv6DNS = _.map(_.reject([this.dns1, this.dns2], function(dns) { return !self.validateIP(dns); }), 
            function(dns) { return dns.trim(); }).join(',');
          _.extend(ipv6_data, {
            ip: this.ip,
            subnetPrefixLen: this.subnetPrefixLen,
            gateway: this.gateway,
            dns: ipv6DNS
          });
        }
        $.ajax({
          url: 'network_config.php',
          type: 'POST',
          dataType: 'json',
          data: {
            type: 'ipv6',
            config: {
              ipv6: ipv6_data
            },
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data) {
            if (data.redirect)
              redirect(data.redirect);

            if (data.error) {
              msgbus.$emit("alert", "Error", data.error.text, 'error');
              return;
            }

            var msg = "settings have been saved successfully. controller is under restart, please reload page later.";
            //stop trying to redirect to new IPv6 address, since interface param is missing, it will not work
            //if (self.method == 'manual')
            //  msg = msg + " redirecting to new address ...";
            msgbus.$emit("alert", "Done", msg, 'success', 10000); 
            /*
            if (self.method == 'manual')
              _.delay(function() {
                var l = window.location;
                var url = l.href;
                window.location.assign(url.replace(l.hostname, '['+self.ip+']'));
              }, 10000);
              */
          },
          error: function(data, status) {
            msgbus.$emit("alert", "Error", "failed to save settings.", 'error'); 
          },
          complete: function(data, status) {
            self.getSpinner().stop();
          },
        });
      }
    }
  }; //}}}1

  //create&init root object
  var app = new Vue({ //{{{1
    el: '#network_settings_content',
    data: function() {
      let sections = [];
      if (window.iot_platform == 'FS') 
        sections = [
          {sec_id: 'general', name: 'General'},
          {sec_id: 'security', name: '802.1x Security'},
          {sec_id: 'ipv6', name: 'IPv6'}
        ];
      else if (window.iot_platform == 'FW')
        sections = [
          {sec_id: 'security', name: '802.1x Security'},
        ];

      if (sections.length == 0)
        return {};

      return {
        sections: sections,
        active_sec_id: sections[0].sec_id
      }
    },
    methods: {
      isSectionActive: function(sec) { return sec.sec_id == this.active_sec_id; }
    },
    components: {
      'general': generalComp,
      'security': securityComp,
      'ipv6': ipv6Comp,
      'alert': alertComp,
      'file-selector': fileSelectorComp,
    }
  }); //}}}1

});

// vim: foldmethod=marker
