// vim: ts=2 sw=2 foldmethod=marker
$(function() {

  var bacnetOnly = false;

  // general functions  {{{1
  function uniqueId(prefix, existingIds) {
    var newId = '';
    var suffixNum = 0;
    do {
      newId = prefix+ "_" + suffixNum.toString();
      ++suffixNum;
    } while (_.contains(existingIds, newId));
    return newId;
  }
  function uniqueSlotId(prefix) {
    return prefix + (new Date()).getTime().toString() + "_" + _.random(0, 10000).toString();
  }

  function uniqueName(prefix, existingNames) {
    var newName = '';
    var suffixNum = 0;
    do {
      newName = prefix+ "_" + suffixNum.toString();
      ++suffixNum;
    } while (_.contains(existingNames, newName));
    return newName;
  }

  function mapObject(obj, processor) {
    var keys = _.keys(obj)
    var results = {}, currentKey;
    for(var index=0; index<keys.length; ++index) {
      currentKey = keys[index];
      results[currentKey] = processor(obj[currentKey], currentKey, obj);
    }
    return results;
  }

	function findPos(obj) {
		var curleft = curtop = 0;
		if (obj.offsetParent) {
			do {
				curleft += obj.offsetLeft;
				curtop += obj.offsetTop;
			} while (obj = obj.offsetParent);
		}
    return [curleft, curtop];
	}

  function getShortTypeName(sedonaNodeData) {
    var type = null;
    if (_.has(sedonaNodeData, 'type'))
      type = sedonaNodeData.type;

    var shortName = null;
    if (type == "easyioFGBacnet::BACnetClientBV")
      shortName = 'BV';
    else if (type == "easyioFGBacnet::BACnetClientBI")
      shortName = 'BI';
    else if (type == "easyioFGBacnet::BACnetClientBO")
      shortName = 'BO';
    else if (type == "easyioFGBacnet::BACnetClientAV")
      shortName = 'AV';
    else if (type == "easyioFGBacnet::BACnetClientAI")
      shortName = 'AI';
    else if (type == "easyioFGBacnet::BACnetClientAO")
      shortName = 'AO';
    return shortName;
  }

  const brokerTypeList = [
    'gcp-iot-core',
    'mqtt',
    'aws-iot-core',
  ];
  function brokerTypeLabel(type) {
    const typeLabelMapping = {
      'gcp-iot-core': 'Google Cloud',
      'mqtt': 'General MQTT',
      'aws-iot-core': 'AWS IoT Core',
    };
    if (_.has(typeLabelMapping, type))
      return typeLabelMapping[type];
    else
      return type;
  }

  function isNameValid(name) {
    return name && /^[-_a-zA-Z0-9]{1,32}$/.test(name);
  }

  function isTopicPathValid(path) {
    return path && /^[-_/a-zA-Z0-9]{1,256}$/.test(path);
  }
  // }}}1

  Vue.component("service-status", { // {{{1
    template: '<div class="pull-right"><span class="label" :class="status_class">{{ label }}</span></div>',
    data: function() {
      return {
        status: '',
        label: 'unavailable',
      };
    },
    computed: {
      status_class: function() {
        if (this.status.length > 0)
          return "label-" + this.status;
        else
          return '';
      },
    },
    mounted: function() {
      this.updateServiceStatus = _.debounce(_.bind(this.doUpdateServiceStatus, this), 10000),
      this.doUpdateServiceStatus();
    },
    methods: {
      updateServiceStatus: function() {},
      doUpdateServiceStatus: function() {
        var self = this;
        $.ajax({
          url: 'index.php?action=status',
          method: 'GET',
          dataType: 'json',
          success: function(resp) {
            if (resp.redirect)
              redirect(resp.redirect);

            if (!resp.service_status) {
              self.status = "";
              self.label = 'unavailable';
              return;
            }

            var resp_status = resp.service_status;
            if (resp_status == 'started') {
              self.status = "success";
              self.label = 'started';
            } else if (resp_status == 'stopped') {
              self.status = "warning";
              self.label = 'stopped';
            } else if (resp_status == 'error') {
              self.status = "important";
              self.label = 'error';
            } else {
              self.status = "";
              self.label = 'unavailable';
            }
          },
          error: function(data, status) {
            self.status = '';
            self.label = 'unavailable';
            console.warn("get error: " + data);
          },
          complete: function() {
            self.updateServiceStatus();
          },
        });
      },
    },
  }); // }}}1

  Vue.component("modal-dialog", { //{{{1
    template: "#modal_dialog_template",
    props: ['title'],
    data: function() {
      return {
      };
    },
    mounted: function() {
      let self = this;
      this.$el.onshow = function() { $(self.$el).find("input").focus(); };
    },
    methods: {
      onSubmit: function() {
        this.$emit("submitChange");
      },
    },
  }); //}}}1

  Vue.component("sortable-table-header", { //{{{1
    template: "#sortable_table_header_template",
    props: ["sortField", "fieldName"],
    data: function() {
      return {
        ascendingSort: true,
      };
    },
    watch: {
      ascendingSort: function(newVal, oldVal) {
        this.$emit("sortBy", this.fieldName, newVal);
      },
    },
  }); //}}}1

  Vue.filter('capitalize', function (value) { //{{{1
    if (!value) return '';
    value = value.toString();
    return value.charAt(0).toUpperCase() + value.slice(1);
  }); //}}}1

  //define components 
  let gcpIoTCoreComp = { //{{{1
    template: '#gcp_iot_core_template',
    props: ['brokerName', 'configData', 'nameError'],
    data: function() {
      let _data = {
        name: this.brokerName,
        config: _.clone(this.configData),
      };
      if (!_.has(_data.config, "region"))
        _data.config['region'] = 'us-central1';
      return _data;
    },
    computed: {
      name_error: function() {
        if (this.nameError.length > 0)
          return this.nameError;

        return isNameValid(this.name) ? "" : "Invalid Broker Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      registry_id_error: function() {
        return this.config.registry_id.length > 0 ? "" : "Invalid Registry ID";
      },
      project_id_error: function() {
        return this.config.project_id.length > 0 ? "" : "Invalid Project ID";
      },
      device_id_error: function() {
        return this.config.device_id.length > 0 ? "" : "Invalid Device ID";
      },
      region_error: function() {
        if (_.has(this.config, 'region'))
          return this.config.region.length > 0 ? "" : "Invalid Region";
        else
          return "";
      },
      ca_file_error: function() {
        return this.config.ca_file.length > 0 ? "" : "Invalid CA File";
      },
      cert_file_error: function() {
        return this.config.cert_file.length > 0 ? "" : "Invalid Certificate File";
      },
      key_file_error: function() {
        return this.config.key_file.length > 0 ? "" : "Invalid Key File";
      },
      events_interval_error: function() {
        return _.isNumber(this.config.events_interval) && this.config.events_interval > 0 ? "" : "Invalid Update Interval";
      },
    },
    methods: {
      changeFile: function(title, oldval, callback) { 
        var inst = this.$root.$refs['file_selector_inst'];
        if (!inst)
          return;

        inst.doModal(title, oldval, callback);
      },
      changeCAFile: function() {
        var self = this;
        this.changeFile("Choose CA File", self.config.ca_file, function(path) {
          self.$set(self.config, 'ca_file', path);
        });
      },
      changeClientCertFile: function() {
        var self = this;
        this.changeFile("Choose Client Cert File", self.config.cert_file, function(path) {
          self.$set(self.config, 'cert_file', path);
        });
      },
      changeKeyFile: function() {
        var self = this;
        this.changeFile("Choose Key File", self.config.key_file, function(path) {
          self.$set(self.config, 'key_file', path);
        });
      },

      onSaveLocal: function() {
        this.$emit("brokerUpdated", this.name, this.config);
      },
      onReset: function() {
        this.$set(this, 'name', this.brokerName);
        this.$set(this, 'config', _.clone(this.configData));
      },
    },
  }; // }}}1

  let mqttComp = { //{{{1
    template: '#mqtt-template',
    props: ['brokerName', 'configData', 'nameError'],
    data: function() {
      return {
        name: this.brokerName,
        config: _.clone(this.configData),
      };
    },
    mounted: function() {
      if (_.has(this.config, 'no_tls'))
        this.tls_enabled = !this.config.no_tls;
    },
    computed: {
      name_error: function() {
        if (this.nameError.length > 0)
          return this.nameError;

        return isNameValid(this.name) ? "" : "Invalid Broker Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      host_error: function() {
        return this.config.mqtt_host.length > 0 ? "" : "Invalid MQTT Host Name";
      },
      port_error: function() {
        return _.isNumber(this.config.mqtt_port) && this.config.mqtt_port > 0 ? "" : "Invalid MQTT Port";
      },
      client_id_error: function() {
        return "";
      },
      user_error: function() {
        return "";
        //return this.config.mqtt_user.length > 0 ? "" : "Invalid User Name";
      },
      passwd_error: function() {
        return "";
        //return this.config.mqtt_passwd.length > 0 ? "" : "Invalid Password";
      },

      ca_file_error: function() {
        return this.config.no_tls || this.config.ca_file.length > 0 ? "" : "Invalid CA File";
      },
      cert_file_error: function() {
        if (this.config.no_tls)
          return "";

        if ( this.config.cert_file.length == 0 && this.config.key_file.length > 0 )
          return "Invalid Certificate File";
        else
          return "";
      },
      key_file_error: function() {
        if (this.config.no_tls)
          return "";

        if ( this.config.key_file.length == 0 && this.config.cert_file.length > 0 )
          return "Invalid Key File";
        else
          return "";
      },

      events_interval_error: function() {
        return _.isNumber(this.config.events_interval) && this.config.events_interval > 0 ? "" : "Invalid Update Interval";
      },

      tls_enabled: {
        get: function() {
          return !this.config.no_tls;
        },
        set: function(newVal) {
          this.$set(this.config, 'no_tls', !newVal);    
        },
      },

    },
    methods: {
      changeFile: function(title, oldval, callback) { 
        var inst = this.$root.$refs['file_selector_inst'];
        if (!inst)
          return;

        inst.doModal(title, oldval, callback);
      },
      changeCAFile: function() {
        if (this.config.no_tls)
          return;

        var self = this;
        this.changeFile("Choose CA File", self.config.ca_file, function(path) {
          self.$set(self.config, 'ca_file', path);
        });
      },
      changeClientCertFile: function() {
        if (this.config.no_tls)
          return;

        var self = this;
        this.changeFile("Choose Client Cert File", self.config.cert_file, function(path) {
          self.$set(self.config, 'cert_file', path);
        });
      },
      changeKeyFile: function() {
        if (this.config.no_tls)
          return;

        var self = this;
        this.changeFile("Choose Key File", self.config.key_file, function(path) {
          self.$set(self.config, 'key_file', path);
        });
      },

      onSaveLocal: function() {
        this.$emit("brokerUpdated", this.name, this.config);
      },
      onReset: function() {
        this.$set(this, 'name', this.brokerName);
        this.$set(this, 'config', _.clone(this.configData));
      },
    },
  }; //}}}1

  let awsIoTCoreComp = { //{{{1
    template: '#aws_iot_core_template',
    props: ['brokerName', 'configData', 'nameError'],
    data: function() {
      return {
        name: this.brokerName,
        config: _.clone(this.configData),
      };
    },
    computed: {
      name_error: function() {
        if (this.nameError.length > 0)
          return this.nameError;

        return isNameValid(this.name) ? "" : "Invalid Broker Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      host_error: function() {
        return this.config.mqtt_host.length > 0 ? "" : "Invalid AWS Endpoint";
      },
      client_id_error: function() {
        return "";
      },
      ca_file_error: function() {
        return this.config.ca_file.length > 0 ? "" : "Invalid CA File";
      },
      cert_file_error: function() {
        if ( this.config.cert_file.length == 0 && this.config.key_file.length > 0 )
          return "Invalid Certificate File";
        else
          return "";
      },
      key_file_error: function() {
        if ( this.config.key_file.length == 0 && this.config.cert_file.length > 0 )
          return "Invalid Key File";
        else
          return "";
      },

      events_interval_error: function() {
        return _.isNumber(this.config.events_interval) && this.config.events_interval > 0 ? "" : "Invalid Update Interval";
      },
    },
    methods: {
      changeFile: function(title, oldval, callback) { 
        var inst = this.$root.$refs['file_selector_inst'];
        if (!inst)
          return;

        inst.doModal(title, oldval, callback);
      },
      changeCAFile: function() {
        if (this.config.no_tls)
          return;

        var self = this;
        this.changeFile("Choose CA File", self.config.ca_file, function(path) {
          self.$set(self.config, 'ca_file', path);
        });
      },
      changeClientCertFile: function() {
        if (this.config.no_tls)
          return;

        var self = this;
        this.changeFile("Choose Client Cert File", self.config.cert_file, function(path) {
          self.$set(self.config, 'cert_file', path);
        });
      },
      changeKeyFile: function() {
        if (this.config.no_tls)
          return;

        var self = this;
        this.changeFile("Choose Key File", self.config.key_file, function(path) {
          self.$set(self.config, 'key_file', path);
        });
      },

      onSaveLocal: function() {
        this.$emit("brokerUpdated", this.name, this.config);
      },
      onReset: function() {
        this.$set(this, 'name', this.brokerName);
        this.$set(this, 'config', _.clone(this.configData));
      },
    },
  }; //}}}1

  let propertyListPanelComp = {  // {{{1
    template: '#property_list_panel_template',
    props: ['objData'],
    data: function() {
      return {
        objDataDomID: {},
        boxOpened: true,
      };
    },
    computed: {
      boxBtnClass: function() {
        if (this.boxOpened)
          return 'icon-folder-open';
        else
          return 'icon-folder-close';
      },
    },
    methods: {
      objDataUniqueDomID: function(objDataPath) {
        if (!_.has(this.objDataDomID, objDataPath))
          this.objDataDomID[objDataPath] = _.uniqueId('data_table_');
        return this.objDataDomID[objDataPath];
      },

      addProperty: function() {
        this.objData.slots.push({
          name: '', value: '', type: 'Str', slotType: 'property', enabled: true, visible: true
        });
      },

      selectAll: function() {
        _.each(this.objData.slots, function(slot) { slot.enabled = true; });
      },
      inverseSelect: function() {
        _.each(this.objData.slots, function(slot) { slot.enabled = !slot.enabled; });
      },
      selectNone: function() {
        _.each(this.objData.slots, function(slot) { slot.enabled = false; });
      },
      deleteSelectedSlot: function() {
        this.objData.slots = this.objData.slots.filter(function(s) {
          return !s.enabled;
        });
      },
    },
  }; // }}}1

  let nameReplacementPreprocessorComp = { // {{{1
    template: '#name_replacement_preprocessor_template',
    props: ['payload', 'enabled'],
    data: function() {
      return {
        tooltip: "change slot's label based on its name and select it",
      };
    },
    created: function() {
      _.defaults(this.payload, {old_name: '', new_name: ''});
    },
    methods: {
      process: function(slot) {
        if (!slot)
          return;

        if (slot.name === this.payload.old_name)
        {
          slot.label = this.payload.new_name;
          slot.enabled = true;
        }
      },
    },
  }; //}}}1

  let preselectPreprocessorComp = { // {{{1
    template: '#preselect_preprocessor_template',
    props: ['payload', 'enabled'],
    data: function() {
      return {
        tooltip: "select slot when its name match one of names specified here",
      };
    },
    created: function() {
      _.defaults(this.payload, {name_list: []});
    },
    methods: {
      process: function(slot) {
      },
    },
  }; // }}}1

  let preprocessorListComp = { // {{{1
    template: '#preprocessor_list_template',
    data: function() {
      return {
        type_list: [
          {label: 'Name Replacement', icon: 'icon-font', component: 'name-replacement-preprocessor-comp'},
          // {label: 'Preselection', icon: 'icon-check', component: 'preselect-preprocessor-comp'},
        ],
        instance_list: [],
      };
    },
    methods: {
      addPreprocessorElem: function(type) {
        this.instance_list.push({
          enabled: true,
          component: type.component,
          payload: {},
        });
      },

      iconFromComp: function(component) {
        var type = _.find(this.type_list, function(type) { return type.component === component; });
        var icon = null;
        if (type) 
          icon = type.icon;
        if (!icon)
          icon = 'icon-wrench';
        return icon;
      },

      initData: function(data) {
        if (data && _.has(data, 'preprocessor_list'))
          this.initPreprocessorList(data.preprocessor_list);
      },
      initPreprocessorList: function(inst_list) {
        var self = this;
        this.instance_list.splice(0, this.instance_list.length);
        inst_list.forEach(function(inst) {
          var isCompValid = _.any(self.type_list, function(type) {
            return type.component === inst.component;
          });
          if (isCompValid) {
            if (_.isString(inst['enabled']))
              inst['enabled'] = inst['enabled'] == 'true';

            self.instance_list.push(inst);
          }
        });
      },
      
      toJson: function() {
        return JSON.parse(JSON.stringify(this.instance_list));
      },

      process: function(slot) {
        if (!slot)
          return;

        _.each(this.$refs.preprocessor, function(preprocessor) {
          if (preprocessor.enabled)
            preprocessor.process(slot);
        });
      },
    },
    components: {
      'name-replacement-preprocessor-comp': nameReplacementPreprocessorComp,
      'preselect-preprocessor-comp': preselectPreprocessorComp,
    },
  }; //}}}1
  
  let deviceDataPanelComp = { //{{{1
    template: '#device_data_panel_template',
    data: function() {
      return {
      };
    },
    deviceRawData: {
    },
    mounted: function() {
      var self = this;
      $(this.$el).find('#deviceDataTree').on('dblclick.jstree', function(event) {
        var tree = $(this).jstree();
        if (tree)
          self.addObjectToSchema();
      }).jstree({
        'plugins': ['wholerow', 'sort'],
        'core' : {
          'themes': {
            'dots': true,
          },
          'multiple': false,
          'dblclick_toggle': false,
          'data': function(node, cb) {
            var url;
            if (node.id === '#') {
              //GOTCHA: for GCP IoT, only expose BACnet data 
              if (bacnetOnly)
                cb.call(this, [{text: 'BACnet', children: true, path: '/EasyIO/BACnetS/BACnetC', state: {'selected':true, 'opened':true} }]);
              else
                cb.call(this, [{text: 'Controller', children: true, path: '/', state: {'selected':true, 'opened':true} }]);
              return ;
            }
            else {
              if (node.original && node.original.path)
                url = node.original.path;
              else {
                cb.call(this, false);
                return;
              }
            }

            self.loadNodeData(url, cb);
          },
        },
      });
      msgbus.$on("focusSedonaObj", this.focusObj);
    },
    methods: {
      jstree: function() {
        return $(this.$el).find("#deviceDataTree").jstree(true);
      },
      loadNodeData: function(nodeURL, cb) {
        var self = this;
        var baseURL = '../../app/data_api.php';

        let url = '/app/objects';
        if (nodeURL == '/')
          url = url + nodeURL + '*';
        else
          url = url + nodeURL + '/*';
        $.ajax({
          url: baseURL,
          method: 'GET',
          dataType: 'json',
          data: {
            url: url,
          },
          success: function(data, status, jq) {
            if (data.redirect)
              redirect(data.redirect);

            if (!data.response || !data.response.data) {
              console.warn("invalid response data format");
              return;
            }

            if (!_.isArray(data.response.data) 
              || data.response.data.length <= 0) {
              console.warn("invalid data format");
              return;
            }
            else {
              if (!_.isArray(data.response.data[0]))
                data.response.data = [data.response.data];
            }

            var treeData = self.convertData(data.response.data);
            if (treeData)
              cb.call(this, treeData);
          },
        });
      },
      convertData: function(data) {
        var treeData = [];
        for(var i=0; i<data.length; ++i) {
          for(var j=0; j<data[i].length; ++j) {
            var objData = data[i][j];
            var path = objData.path;
            var name = this.getFriendlyObjectName(objData);

            if (bacnetOnly) {
              //GOTCHA: add BACnet deviceId if possible
              //        this will be used to access data from BACnet shared memory
              //        this should be deleted when we don't get sedona data from
              //        shared memory 
              var deviceId = this.getBACnetDeviceId(path);
              if (deviceId)
                objData['deviceId'] = deviceId;
            }

            var hasChildren = _.has(objData, 'childNum') && parseInt(objData.childNum)>0;
            var icon = 'jstree-folder';
            if (!hasChildren) icon = 'jstree-file';
            var node = {id: path, path: path, text: name, children: hasChildren, icon:icon, raw: objData};
            treeData.push(node);
          }
        }
        return treeData;
      },
      getParentPath: function(path) {
        if (!path)
          return null;

        var parts = path.split("/");
        parts.pop();
        return parts.join("/");
      },
      getFriendlyObjectName: function(sedonaNodeData) {
        //GOTCHA: since there is no object type info returned from easyioCpt
        //kit, we check the slots here to detect BACnet information
        var bacnetDeviceName = '';
        var bacnetDeviceID = null;
        var bacnetPointName = ''
        var bacnetPointID = null;
        for(var i=0; i<sedonaNodeData.slots.length; ++i) {
          var slot = sedonaNodeData.slots[i];
          if (slot.name == 'deviceId')
            bacnetDeviceID = slot.value;
          else if (slot.name == 'deviceName')
            bacnetDeviceName = slot.value;
          else if (slot.name == 'objectId')
            bacnetPointID = slot.value;
          else if (slot.name == 'pointName')
            bacnetPointName = slot.value;
        }

        if (bacnetDeviceID && bacnetDeviceName.length > 0) {
          return bacnetDeviceName + ' [' + bacnetDeviceID + ']';
        }
        else if (bacnetPointID && bacnetPointName.length > 0) {
          var shortName = getShortTypeName(sedonaNodeData);
          if (shortName)
            return bacnetPointName + ' [' + shortName + ':' + bacnetPointID + ']';
          else
            return bacnetPointName + ' [' + bacnetPointID + ']';
        }
        else
        {
          var name = _.last(sedonaNodeData.path.split("/"));
          return name;
        }
      },
      getBACnetDeviceId: function(path) {
        var bacnetDeviceId = null;

        var jstree = this.jstree();
        var parentId = this.getParentPath(path);
        var parentNode = jstree.get_node(parentId);
        if (!parentNode)
          return bacnetDeviceId;

        if (_.has(parentNode.original, 'raw') && parentNode.original.raw && _.has(parentNode.original.raw, 'slots')) {
          _.each(parentNode.original.raw.slots, function(slot) {
            if (slot.name == 'deviceId')
              bacnetDeviceId = slot.value;
          });
        }
        return bacnetDeviceId;
      },

      addObjectToSchema: function() {
        var objs = _.map(this.jstree().get_selected(true), function(node) {
          if (!node.original)
            return null;
          return node.original;
        });
        this.$emit('objectsToSchema', _.compact(objs)); 
      },

      focusObj: function(path) {
        var jstree = this.jstree();
        var node = jstree.get_node(path);
        if (node) {
          var curSelected = jstree.get_selected(false);
          jstree.deselect_node(curSelected);
          jstree.select_node(node);

          //scroll to the selected node, ref: https://github.com/vakata/jstree/issues/519
          let jstreeDom = document.getElementById("deviceDataTree");
          jstreeDom.scrollTop = findPos( document.getElementById(path) )[1] - jstreeDom.offsetHeight/2;
        }
      },
    },
  }; // }}}1

  let nodeSchemaPanelComp = { //{{{1
    template: '#node_schema_panel_template',
    inject: ['getGConfig'],
    data: function() {
      return {
        nodeID: null,
        objDataList: [],
        showAll: true,
        selectedSlotUID: null,
        editMode: false,
        errorSlotUID: null,
      };
    },
    mounted: function() {
      this.updateVisibilityState();
      msgbus.$on('propRenamed', _.bind(this.onPropRenamed, this));
    },
    computed: {
      commonBindingPath: function() {
        if (this.objDataList.length < 1)
          return "";

        var pathPartList = _.map(this.objDataList, function(objData) { return objData.path.split("/"); });
        var commonLen = 0;
        _.any(_.range(pathPartList[0].length-1), function(index) {
          var part = null;
          var allIncludesPart = _.all(pathPartList, function(partList) {
            if (_.isNull(part)) {
              part = partList[index];
              return true
            } else
              return partList[index] == part;
          });
          if (allIncludesPart)
            commonLen = index + 1;
          return !allIncludesPart;
        });

        if (commonLen < 2)
          return "";
        else
          return _.first(pathPartList[0], commonLen).join("/");
      },
    },
    watch: {
      showAll: function(newVal, oldVal) {
        this.updateVisibilityState();
      },
      selectedSlotUID: function(newVal, oldVal) {
        var self = this;
        var objData = _.find(this.objDataList, function(objData) {
          return _.any(objData.slots, function(slot) {
            return slot.uid == self.selectedSlotUID;
          });
        });
        if (objData)
          msgbus.$emit("focusSedonaObj", objData.path);
      },
    },
    methods: {
      bindingTooltip: function(slotData) {
        if (bacnetOnly)
          return slotData.path.replace('/EasyIO/BACnetS/BACnetC/', '/BACnet/');
        else
          return slotData.path;
      },
      shortBindingPath: function(slotData) {
        var result = slotData.path;
        var commonLen = this.commonBindingPath.split("/").length;
        if (commonLen > 0) {
          var parts = slotData.path.split("/");
          result = _.last(parts, parts.length-commonLen).join("/");
        }
        return result;
      },
      scrollToBottom: function() {
        var elem = this.$el.querySelector('.scrollable-content');
        $(elem).animate({scrollTop: elem.scrollHeight}, 300);
      },
      scrollToTop: function() {
        var elem = this.$el.querySelector('.scrollable-content');
        $(elem).animate({scrollTop: 0}, 300);
      },
      scrollToSlot: function(slotId) {
        var self = this;
        _.any(this.$refs.slotPanel, function(slotPanel) {
          if (slotPanel.slotData.uid == slotId) {
            slotPanel.highlight();
            return true;
          } else
            return false;
        });
      },

      compType: function(objData) {
        if (_.has(objData, "compType") && objData.compType == "propertyList")
          return "property-list-panel-comp";
        else
          return "comp-obj-data-panel-comp";
      },

      addObjects: function(objs) {
        //clear old data
        this.selectedSlotUID = null;
        this.objDataList.splice(0, this.objDataList.length);

        for(var i=0; i<objs.length; ++i) {
          var obj = objs[i];
          if (!obj.raw)
            continue;

          //prevent to add duplicate objData
          if (_.any(this.objDataList, function(objData) {
            return objData.path === obj.path;
          })) {
            console.info("device data is already added: " + obj.path);
            continue;
          }

          var objData = {};
          var objPath = obj.path;
          objData['path'] = objPath;

          var slots = obj.raw.slots;
          if (bacnetOnly && _.has(obj, 'raw')) {
            objData['bacnetPtType'] = getShortTypeName(obj.raw);
            //GOTCHA: copy over BACnet deviceId 
            //        should delete this hack when don't get data from BACnet
            //        shared memory 
            if (_.has(obj.raw, 'deviceId'))
              objData['deviceId'] = obj.raw.deviceId;
            _.any(slots, function(slot) {
              if (slot.name == 'objectId') {
                objData['objectId'] = slot.value;
                return true;
              } else 
                return false;
            });
          }

          objData['slots'] = [];
          for(var j=0; j<slots.length; ++j) {
            var slot = slots[j];

            if (slot.slotType != 'property')
              continue;

            //FIXME: now we only can fetch BACnet object's data from shared
            //memory, and from shared memory, only following slots' data can be
            //fetched, so only display these slots to user, otherwise user will
            //become confused. this filter need to be removed when we support
            //fetching data from the data layer later
            if (bacnetOnly && !_.contains(['objectId', 'pointName', 'presentValue'], slot.name))
              continue;

            var slotCopy = {};
            slotCopy['uid'] = uniqueSlotId("#slot_");
            slotCopy['name'] = slot.name; //keep the sedona original slot name
            slotCopy['label'] = slot.name; //used for display
            slotCopy['path'] = objPath + "." + slot.name;
            slotCopy['type'] = slot.type;
            slotCopy['enabled'] = false;
            slotCopy['visible'] = true;
            objData['slots'].push(slotCopy);
          }
          this.objDataList.push(objData);
          this.$emit("objectAdded", objData);
        }
        var self = this;
        _.any(this.objDataList, function(objData) {
          return _.any(objData.slots, function(slot) {
            if (slot.enabled) {
              self.selectedSlotUID = slot.uid;
              return true;
            } else 
              return false;
          });
        });

        if (!this.selectedSlotUID && this.objDataList.length>0
            && this.objDataList[0].slots.length > 0)
          this.selectedSlotUID = this.objDataList[0].slots[0].uid;
      },

      //TODO: add propertyList to DataSchemaTree and add slotUID
      createPropertyList: function() {
        if (_.any(this.objDataList, function(objData) {
          return objData.compType === 'propertyList';
        }))
          return;

        //add default property list panel
        this.objDataList.push({
          path: 'PropertyList',
          compType: 'propertyList',
          slots: [
            { name: '', value: '', type: 'Str', 
              slotType: 'property', enabled: true,
              visible: true
            },
          ],
        });
        this.scrollToBottom();
      },
      
      cloudNodeSiblingNames: function() {
        var names = [];
        var cloudTree = $("#dataSchemaTree").jstree(true);
        if (!cloudTree)
          return names;

        var objNode = null;
        if (cloudTree.get_type(this.nodeID) == 'property')
          objNode = cloudTree.get_node(cloudTree.get_parent(this.nodeID));
        else
          objNode = cloudTree.get_node(this.nodeID);

        var self = this;
        return _.compact(_.map(objNode.children, function(child) {
          if (child == self.nodeID)
            return "";
          else
            return cloudTree.get_text(child);
        }));
      },
      currentBrokerType: function(propNode) {
        var cloudTree = $("#dataSchemaTree").jstree(true);
        if (!cloudTree)
          return null;

        var node = cloudTree.get_node(this.nodeID);
        if (!node)
          return null;

        // 1. find topic name 
        while(node.id != "#" && node.type != "topic") {
          node = cloudTree.get_node(node.parent);
        }

        if (node.type != "topic")
          return null;

        // 2. find topic object
        var topic = _.find(gConfig.topicList, function(topic) {
          return topic.name == node.text;
        });
        if (!topic)
          return null;

        // 3. find broker object
        var brokerName = topic.broker;
        var broker = _.find(gConfig.brokerList, function(broker) {
          return brokerName == broker.name;
        });
        if (!broker)
          return null;
        return broker.type;
      },
      validateInputs: function() {
        var self = this;
        var curBrokerType = this.currentBrokerType();
        var siblingNames = this.cloudNodeSiblingNames();
        var isValid = _.all(this.objDataList, function(objData) {
          return _.all(objData.slots, function(slot) {
            if (!slot.enabled)
              return true;

            if (_.contains(siblingNames, slot.label)) {
              self.errorSlotUID = slot.uid;
              return false;
            } else {
              if (curBrokerType == "gcp-iot-core") {
                // for gcp-iot-core, the property's name must be
                // 'present_value', more details:
                // https://github.com/grafnu/daq/blob/pubber/schemas/udmi/README.md
                if (slot.label != "present_value") {
                  self.errorSlotUID = slot.uid;
                  return false;
                } else 
                  return true;
              } else 
                return true;
            }
          });
        });

        if (isValid)
          this.errorSlotUID = null;
        return isValid;
      },
      saveNodeSchema: function() {
        if (!this.nodeID)
          return;

        var self = this;
        _.each(this.objDataList, function(objData) {
          _.each(objData.slots, function(slot) {
            slot.enabled = slot.uid == self.selectedSlotUID;
          });
        });
        if (this.validateInputs()) {
          var objDataList = JSON.parse(JSON.stringify(this.objDataList))
          this.$emit('nodeSchemaUpdated', this.nodeID, objDataList);
        }
        else
          console.warn("invalid input");
      },

      updateNodeSchema: function(nodeID, objDataList, slotUID) {
        var self = this;
        this.nodeID = nodeID;
        this.selectedSlotUID = slotUID;
        this.objDataList.splice(0, this.objDataList.length);
        this.editMode = !!slotUID;

        if (objDataList) {
          objDataList.forEach(function(objData) {
            self.objDataList.push(objData);
          });
        }

        this.updateVisibilityState();
        
        // if (this.selectedSlotUID)
        //   //defer to allow UI rendered
        //   _.defer(_.bind(this.scrollToSlot, this, this.selectedSlotUID));
        // else
        //   this.scrollToTop();
      },

      updateVisibilityState: function() {
        if (!this.showAll) {
          var slotNameList = _.reject(_.map(this.objDataList, function(objData) {
            if (_.has(objData, 'compType') && objData.compType == 'propertyList')
              return [];

            return _.map(objData.slots, function(slot) { return slot.name; });
          }), _.isEmpty);
          var commonSlotNames = _.intersection.apply(_, slotNameList);

          _.each(this.objDataList, function(objData) {
            if (_.has(objData, 'compType') && objData.compType == 'propertyList')
              return;
            _.each(objData.slots, function(slot) {
              slot.visible = _.contains(commonSlotNames, slot.name);
            });
          });
        }
        else {
          _.each(this.objDataList, function(objData) {
            if (_.has(objData, 'compType') && objData.compType == 'propertyList')
              return;
            _.each(objData.slots, function(slot) {
              slot.visible = true;
            });
          });
        }
      },

      onPropRenamed: function(slotUID, slotLabel) {
        if (this.selectedSlotUID != slotUID)
          return;

        _.any(this.objDataList, function(objData) {
          return _.any(objData.slots, function(slot) {
            if (slot.uid == slotUID) {
              slot.label = slotLabel;
              return true;
            } else 
              return false;
          });
        });
      },

      onSlotDataChanged: function(index) {
        var curSlotPanel = this.$refs.slotPanel[index];
        var slotLabel = curSlotPanel.slotData.label;
        if (!curSlotPanel.slotData.enabled) {
          curSlotPanel.markError(false);
          return;
        }
        for(var i=0; i<this.$refs.slotPanel.length; ++i) {
          if (i == index)
            continue;

          var slotPanel = this.$refs.slotPanel[i];
          if (slotPanel.slotData.enabled && slotLabel == slotPanel.slotData.label) {
            curSlotPanel.markError(true);
            return;
          }
        }
        curSlotPanel.markError(false);
      },

    },
    components: {
      'property-list-panel-comp': propertyListPanelComp,
    },
  }; // }}}1

  let dataSchemaPanelComp = { // {{{1
    template: '#data_schema_panel_template',
    inject: ['getTopicList', 'getGConfig'],
    data: function() {
      return {
        spinner: null,

        selectedTopicName: 'NoMQTTTopic',
        availableTopicNames: [],
        newTopicTreeMode: true,
      };
    },
    mounted: function() {
      var self = this;
      $(this.$el).find('#dataSchemaTree').on('select_node.jstree', function(event, data) {
        if (!data.node)
          return;

        self.onNodeSelected(data.node);
      }).on('rename_node.jstree', function(event, data) {
        var jstree = self.jstree();
        if (!data.node || jstree.get_type(data.node) != 'property')
          return;

        var parentNode = jstree.get_node(data.node.parent);
        _.any(parentNode.original['objDataList'], function(objData) {
          return _.any(objData.slots, function(slot) {
            if (slot.enabled && slot.uid === data.node.id) {
              slot.label = data.text;
              msgbus.$emit("propRenamed", slot.uid, slot.label);
              return true;
            } else 
              return false;
          });
        });
      }).on('delete_node.jstree', function(event, data) {
        var jstree = self.jstree();
        if (!data.node)
          return;

        var parentNode = jstree.get_node(data.parent);
        if (jstree.get_type(data.node) != 'property') {
          jstree.select_node(parentNode);
          return;
        }

        parentNode.original.objDataList = _.reject(parentNode.original.objDataList, function(objData) {
          return _.any(objData.slots, function(slot) {
            return (slot.enabled && slot.uid === data.node.id);
          });
        });
        jstree.select_node(parentNode);
      }).jstree({
        'core': {
          'multiple': false,
          'check_callback': function(op, node, parent, pos, more) {
            if (op == 'rename_node') 
              return self.validateNewName(node, pos);
            else
              return true;
          },
          // 'data' : [{'id': '#device', 'text': 'device', 'type': 'object', 'parent': '#', 'state': {'selected': true, 'opened': true} }],
        },
        'plugins': ['wholerow', 'dnd', 'unique', 'types'],
        'checkbox': {
          'whole_node': true, 
          'keep_selected_style': true,
          'three_state': false,
        },
        'types': {
          '#': {
            'valid_children': ['topic'],
          },
          'topic': {
            'valid_children': ['object', 'property'],
            'icon': './img/topic.png',
          },
          'object': {
            'valid_children': ['object', 'property']
          },
          'property': {
            'valid_children': [],
            'icon': './img/property.png',
          },
        },
        'dnd': {
          is_draggable: function(nodes, event) {
            return !_.any(nodes, function(node) {
              return (node.type == 'property');
            });
          },
        },
      });
      msgbus.$on("topicDeleted", _.bind(this.onTopicDeleted, this));
      msgbus.$on("topicRenamed", _.bind(this.onTopicRenamed, this));
    },
    methods: {
      jstree: function() {
        return $(this.$el).find('#dataSchemaTree').jstree(true);
      },
      getSpinner: function() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },
      initData: function(data) {
        var jstree = this.jstree();
        if (data) {
          if (!_.has(data, 'jstree_data')) {
            msgbus.$emit("alert", "Error", "Invalid tree data.", 'error'); 
            return;
          }

          var jstree_data = data.jstree_data;
          if (_.isArray(jstree_data))
            _.each(jstree_data, function(a_tree_data) {
              jstree.create_node('#', a_tree_data);
            });
          else
            jstree.create_node('#', jstree_data);
          this.onNodeSelected(_.last(jstree.get_selected(true)));
        }
        else {
          var init_root_data = {'id': '#mqtt_topic_0', 'text': 'NoMQTTTopic', 'type': 'topic', 'parent': '#', 'state': {'selected': false, 'opened': true} }
          jstree.create_node('#', init_root_data);
          var init_device_data = {'id': '#device_1', 'text': 'device', 'type': 'object', 'parent': '#mqtt_topic_0', 'state': {'opened': true} }
          var device_node = jstree.create_node('#mqtt_topic_0', init_device_data);
          jstree.select_node(device_node);
        }
      },
      newNode: function() {
        var jstree = this.jstree();
        var curNode = _.last(jstree.get_selected());
        if (!curNode)
          return;

        //always select a object type node as parent
        if (jstree.get_type(curNode) == 'property')
          curNode = jstree.get_parent(curNode);

        var newId = jstree.create_node(curNode, {'type': 'object'});
        jstree.deselect_all();
        jstree.select_node(newId);
        jstree.edit(newId, "new_node");
      },
      pickTopic: function() {
        var jstree = this.jstree();
        var rootNode = jstree.get_node('#');
        var usedTopicList = _.map(rootNode.children, function(childId) { 
          return jstree.get_text(childId);
        });

        this.availableTopicNames = _.difference(
          _.map(this.getTopicList(), function(topic) { return topic.name; }),
          usedTopicList
        );
        this.selectedTopicName = _.first(this.availableTopicNames);

        $(this.$el).find(".modal").modal('show');
      },
      newTopicTree: function() {
        this.newTopicTreeMode = true;
        this.pickTopic();
      },
      createTopicTreeNode: function() {
        $(this.$el).find(".modal").modal('hide');

        if (!this.selectedTopicName)
          return;

        var jstree = this.jstree();
        var rootNode = jstree.get_node('#');
        var newId = uniqueId('#mqtt_topic', rootNode.children);

        var initRootData = {'id': newId, 'text': this.selectedTopicName, 'type': 'topic', 'parent': '#', 'state': {'selected': true, 'opened': true} }
        jstree.create_node('#', initRootData);
        jstree.deselect_all();
        jstree.select_node(newId);
      },
      changeTopicTreeNode: function() {
        $(this.$el).find(".modal").modal('hide');

        if (!this.selectedTopicName)
          return;

        var jstree = this.jstree();
        var selected = jstree.get_selected();
        if (selected.length > 0)
          selected = _.last(selected);

        if (jstree.get_type(selected) != 'topic')
          return;

        jstree.rename_node(selected, this.selectedTopicName);  
      },
      editNode: function() {
        var jstree = this.jstree();
        var selected = jstree.get_selected();
        if (selected.length > 0)
          selected = _.last(selected);

        if (jstree.get_type(selected) == 'topic') {
          this.newTopicTreeMode = false;           
          this.pickTopic();
        } else
          jstree.edit(selected);
      },
      removeNode: function() {
        var jstree = this.jstree();
        jstree.delete_node(jstree.get_selected());
      },

      brokerType: function(node) {
        var jstree = this.jstree();
        // 1. find topic name 
        while(node.id != "#" && node.type != "topic") {
          node = jstree.get_node(node.parent);
        }

        if (node.type != "topic")
          return null;

        var gConfig = this.getGConfig();
        // 2. find topic object
        var topic = _.find(gConfig.topicList, function(topic) {
          return topic.name == node.text;
        });
        if (!topic)
          return null;

        // 3. find broker object
        var brokerName = topic.broker;
        var broker = _.find(gConfig.brokerList, function(broker) {
          return brokerName == broker.name;
        });
        if (!broker)
          return null;
        return broker.type;
      },

      validateNewName: function(node, new_name) {
        var jstree = this.jstree();
        var n = jstree.get_node(node);
        if (this.brokerType(n) != 'gcp-iot-core')
          return true;
        
        if (n.type == 'property')
          return new_name == 'present_value';
        else if (n.type == 'object')
          return /^[a-z][a-z0-9]*(_[a-z0-9]+)*$/.test(new_name);
        else
          return true;
      },

      updateModelData: function(node) {
        var jstree = this.jstree();
        if (jstree.get_type(node) != 'object' && jstree.get_type(node) != 'topic')
          return;

        var self = this;
        if (_.has(node.original, 'objDataList')) {
          var propNodeIds = _.select(node.children, function(child) {
            return jstree.get_type(child) == 'property';
          });
          node.original.objDataList = _.reject(node.original.objDataList, function(objData) {
            return !_.any(objData.slots, function(slot) {
              return slot.enabled && _.includes(propNodeIds, slot.uid);
            });
          });
        }

        _.each(_.select(node.children, function(child) {
          return jstree.get_type(child) == 'object';
        }), function(child) {
          self.updateModelData(jstree.get_node(child));
        });
      },

      onNodeSelected: function(node) {
        if (!node)
          return;

        var jstree = this.jstree();

        var slotUID = null;
        var objDataList = [];
        if (jstree.get_type(node) == "property") {
          slotUID = node.id;
          var objNode = jstree.get_node(node.parent);

          _.any(objNode.original['objDataList'], function(objData) {
            if (_.any(objData.slots, function(slot) {
              return slot.enabled && slot.uid == slotUID;
            })) {
              var copy = _.clone(objData);
              copy.slots = _.map(copy.slots, function(slot) {
                return _.clone(slot);
              });
              objDataList.push(copy);
              return true;
            }
            else 
              return false;
          });
        }

        this.$emit("nodeChanged", node.id, objDataList, slotUID);
      },

      addProperty: function(nodeID, slotData) {
        var jstree = this.jstree();
        if (jstree.get_type(nodeID) == 'property')
          nodeID = jstree.get_parent(nodeID);

        var node = jstree.get_node(nodeID);
        if (!node) {
          console.warn("can not find node by ID: " + nodeID);
          return false;
        }

        if (jstree.get_node(slotData.uid)) {
          console.info("node alreay exists");
          return false;
        }

        var propData = {'id': slotData.uid, 'text': slotData.label, 'type': 'property', 'parent': nodeID, 'state': {'selected': false, 'opened': false} }
        if (!jstree.create_node(node, propData)) {
          //if failed, then most likely the prop'name is not unique, let's
          //generate a unique name now
          var childrenNames = _.map(node.children, function(child) { return jstree.get_text(child); });
          slotData.label = uniqueName(slotData.label, childrenNames);
          propData.text = slotData.label;
          if (!jstree.create_node(node, propData)) {
            console.warn("can not create property node even with a unique name");
            return false;
          }
        }

        if (!jstree.is_open(node))
          jstree.open_node(node);

        return true;
      },

      syncNodeSchema: function(nodeID, objDataList) {
        var self = this;
        var jstree = this.jstree(); 
        var node = jstree.get_node(nodeID);
        if (node.type == 'object' || node.type == 'topic') {
          if (!_.has(node.original, 'objDataList'))
            node.original['objDataList'] = [];

          _.each(objDataList, function(objData) {
            _.any(objData.slots, function(slot) {
              if (slot.enabled) {
                if (self.addProperty(nodeID, slot))
                  node.original['objDataList'].push(objData);
                return true;
              } else
                return false;
            });
          });
        } else if (node.type == 'property') { //update property node
          var objNode = jstree.get_node(node.parent);
          if (!_.has(objNode.original, 'objDataList'))
            objNode.original['objDataList'] = [];

          //remove old objData first 
          objNode.original.objDataList = _.reject(objNode.original.objDataList, function(objData) {
            return _.any(objData.slots, function(slot) {
              return slot.enabled && slot.uid == node.id;
            });
          });

          //update objData and jstree node
          _.each(objDataList, function(objData) {
            _.any(objData.slots, function(slot) {
              if (slot.enabled) {
                objNode.original.objDataList.push(objData);
                jstree.set_id(node, slot.uid);
                jstree.rename_node(slot.uid, slot.label);
                node = jstree.get_node(slot.uid);
                return true;
              } else
                return false;
            });
          });
          
          this.onNodeSelected(node);
        }
      },

      jstreeJson: function() {
        var self = this;
        var jstree = this.jstree();

        this.updateModelData(jstree.get_node("#"));

        var tree_json = jstree.get_json("#");
        //GOTCHA: must copy objDataList from original to node object, otherwise
        //        it will not be dumped to json later
        if (_.isArray(tree_json))
          _.each(tree_json, function(topic_json) {
            self.mergeSlotData(jstree, topic_json);
          });
        else
          this.mergeSlotData(jstree, tree_json);

        return tree_json;
      },
      mappingList: function(tree_json) {
        if (_.isNull(tree_json))
          tree_json = this.jstreeJson();

        let list = [];
        if (!_.isArray(tree_json)) {
          console.warn("invalid tree json format");
          return list;
        }

        var self = this;
        _.each(tree_json, function(topic_json) {
          var topic = topic_json.text;
          list.push(_.extend({topic: topic}, self.buildMappingData(topic_json)));
        });
        return list;
      },
      mappingJson: function(tree_json) {
        var json = {'topics': [], 'mappings': {}};
        if (!_.isArray(tree_json)) {
          console.warn("invalid tree json format");
          return json;
        }

        var self = this;
        _.each(tree_json, function(topic_json) {
          var topic = topic_json.text;
          json.topics.push(topic);
          _.extend(json.mappings, self.buildMappingData(topic_json));
        });
        return json;
      },
      mergeSlotData: function(jstree, node_json) {
        if (!node_json)
          return;

        var node = jstree.get_node(node_json.id);
        if (_.has(node.original, 'objDataList'))
          node_json['objDataList'] = node.original['objDataList'];

        for(var i=0; i<node_json.children.length; ++i) {
          this.mergeSlotData(jstree, node_json.children[i]);
        }
      },
      buildOneNodeMappingData: function(tree_node) {
        var node_data = {};
        if (!tree_node)
          return node_data;

        if (_.has(tree_node, 'objDataList') && _.isArray(tree_node.objDataList)) {
          for(var i=0; i<tree_node.objDataList.length; ++i) {
            var objData = tree_node.objDataList[i];
            if (!_.has(objData, 'slots') || !_.isArray(objData.slots))
              continue;

            var compType = _.has(objData, 'compType') ? objData.compType : null;
            var slots = objData.slots;

            if (bacnetOnly) {
              //GOTCHA: collect 'objectId' and 'deviceId' value in BACnet network 
              //        this is used to locate the BACnet data point
              //        this should be deleted when we don't get sedona data from
              //        shared memory 
              var bacnetData = {};
              if (_.has(objData, 'deviceId') && _.has(objData, 'objectId')) {
                bacnetData['bacnet'] = objData.deviceId + ":" + objData.objectId;
              }

              if (_.has(objData, 'bacnetPtType'))
                bacnetData['bacnetPtType'] = objData.bacnetPtType; 
            }

            for(var j=0; j<slots.length; ++j) {
              var slot = slots[j];
              if (!slot.enabled)
                continue;

              if (compType == 'propertyList')
                node_data[slot.label] = slot.value;
              else
              {
                node_data[slot.label] = {
                  isLeaf: true,
                  path: slot.path,
                  type: slot.type,
                  // slotType: slot.slotType
                };

                if (bacnetOnly) {
                  //GOTCHA: BACnet hack 
                  //        this should be deleted when we don't get sedona data from
                  //        shared memory 
                  if (!_.isEmpty(bacnetData))
                    _.extend(node_data[slot.label], bacnetData);
                }
              }
            }
          }
        }

        if (_.has(tree_node, 'children') && _.isArray(tree_node.children)) {
          for(var i=0; i<tree_node.children.length; ++i) {
            var child_node = tree_node.children[i];
            var child_data = this.buildOneNodeMappingData(child_node);
            if (!_.isEmpty(child_data))
              node_data[child_node.text] = child_data;
          }
        }
        return node_data;
      },
      buildMappingData: function(tree_json) {
        var device_json = {};
        if (!tree_json)
          return device_json;

        var node_data = this.buildOneNodeMappingData(tree_json);
        if (!_.isEmpty(node_data)) {
          device_json[tree_json.text] = node_data;
        }
        return device_json;
      },

      mockPreviewValue: function(value, key) {
        if (_.isObject(value)) {
          if (_.has(value, 'isLeaf') && value.isLeaf) {
            //prepare mock data for preview
            if (_.contains(['float', 'double'], value.type))
              return 42.0;
            else if (_.contains(['int', 'byte', 'short', 'long'], value.type))
              return 42;
            else if (value.type == 'bool')
              return true;
            else
              return ('value_of_' + key).toUpperCase();
          } else
            return mapObject(value, _.bind(this.mockPreviewValue, this));
        }
        else if (_.isArray(value))
          return _.map(value, _.bind(this.mockPreviewValue, this));
        else
          return value;
      },

      previewJsons: function() {
        var result = [];
        var tree_json = this.jstreeJson();
        if (!tree_json)
          return result;
        var mapping_json = this.mappingJson(tree_json);
        if (!mapping_json || !_.has(mapping_json, 'mappings') || !_.has(mapping_json, 'topics'))
          return result;

        var self = this;
        return _.compact(_.map(mapping_json.topics, function(topic) {
          if (!_.has(mapping_json.mappings, topic))
            return null;
          var mapping = mapping_json.mappings[topic];
          return [topic, mapObject(mapping, _.bind(self.mockPreviewValue, self))];
        }));
      },

      topicNodes: function() {
        let jstree = this.jstree();
        let rootNode = jstree.get_node('#');
        return _.select(rootNode.children, function(childId) { 
          return jstree.get_type(childId) == "topic";
        });
      },
      topicNodeByName: function(name) {
        let jstree = this.jstree();
        let idList = this.topicNodes();
        for(let i=0; i<idList.length; ++i) {
          //TODO: this may changed when we display not only the topic name in jstree node later
          if (jstree.get_text(idList[i]) == name)
            return jstree.get_node(idList[i]);
        }
        return null;
      },
      onTopicDeleted: function(name) {
        let jstree = this.jstree();
        let idList = this.topicNodes();
        for(let i=0; i<idList.length; ++i) {
          if (jstree.get_text(idList[i]) != name)
            continue;
          
          let topicNamePlaceholder = uniqueName("NoMQTTTopic", 
            _.map(idList, function(id) { return jstree.get_text(id); }));
          jstree.set_text(idList[i], topicNamePlaceholder);
          return;
        }
      },
      onTopicRenamed: function(oldName, newName) {
        let jstree = this.jstree();
        let topicNode = this.topicNodeByName(oldName);
        if (topicNode)
          jstree.rename_node(topicNode, newName);
      },
    },
  }; // }}}1

  let schemaPreviewEntryComp = { //{{{1
    template: '<div class="schema-preview-entry"><p><img src="./img/topic.png" alt="MQTT Topic"><strong>{{topic}}</strong></p></div>',
    props: ['topic', 'json'], 
    data: function() {
      return {};
    },
    mounted: function() {
      if (!this.json)
        return;
      this.renderJson();
    },
    watch: {
      json: function(newVal, oldVal) {
        this.renderJson();
      },
    },
    methods: {
      renderJson: function() {
        $(this.$el).find('.renderjson').remove();
        var elem = renderjson.set_show_to_level('all')(this.json);
        this.$el.appendChild(elem);
      },
    },
  };
  let schemaPreviewPanelComp = {
    template: '#schema_preview_panel_template',
    props: ['schema_jsons'],
    data: function() {
      return {};
    },
    components: {
      'schema-preview-entry': schemaPreviewEntryComp,
    },
  };
  //}}}1

  let dataMappingComp = { //{{{1
    template: '#data_mapping_template',
    data: function() {
      return {
        spinner: null,
        active_tab_id: 'mapping_editor',
        tabs: [
          {id: 'mapping_editor', label: 'Message Editor'},
          {id: 'schema_preview', label: 'Preview'},
          {id: 'preprocessor', label: 'Preprocessor'}
        ],
        previewJsons: [],
      };
    },
    mounted: function() {
      window.Split(["#dataSchemaPanel", "#nodeSchemaPanel", "#deviceDataPanel"], {
        sizes: [25, 50, 25],
        minSize: 50,
        direction: "horizontal",
        gutterSize: 6,
      });
    },
    watch: {
      active_tab_id: function(newVal, oldVal) {
        if (newVal == 'schema_preview')
          this.updatePreviewJson();
      },
    },
    methods: {
      getSpinner: function() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },

      addObjectsToSchema: function(objs) {
        this.$refs.nodeSchemaPanel.addObjects(objs);
      },

      onNodeChanged: function(nodeID, objDataList, slotUID) {
        this.$refs.nodeSchemaPanel.updateNodeSchema(nodeID, objDataList, slotUID);
      },

      onNodeSchemaUpdated: function(nodeID, objDataList) {
        this.$refs.dataSchemaPanel.syncNodeSchema(nodeID, objDataList);
      },

      onObjectAdded: function(objData) {
        var preprocessorList = this.$refs.preprocessorListPanel;
        if (preprocessorList && objData && _.has(objData, 'slots')) {
          _.each(objData.slots, function(slot) {
            preprocessorList.process(slot);
          });
        }
      },

      onShowTab: function(event) {
        this.active_tab_id = $(event.target).data('tab-id');
      },

      updatePreviewJson: function() {
        if (!this.$refs.dataSchemaPanel)
          this.previewJsons = [];
        else
          this.previewJsons = this.$refs.dataSchemaPanel.previewJsons();
      },

      getMappingData: function() {
        var tree_json = this.$refs.dataSchemaPanel.jstreeJson();
        if (!tree_json) 
          return;
        return this.$refs.dataSchemaPanel.mappingList(tree_json);
      },
      getConfigData: function() {
        let config_json = {};
        let tree_json = this.$refs.dataSchemaPanel.jstreeJson();
        if (!tree_json) 
          return config_json;
        config_json['jstree_data'] = tree_json;
        config_json['preprocessor_list'] = this.$refs.preprocessorListPanel.toJson();
        return config_json;
      },

      initData: function(data) {
        this.$refs.dataSchemaPanel.initData(data);
        this.$refs.preprocessorListPanel.initData(data);
      },
    },
    components: {
      'device-data-panel': deviceDataPanelComp,
      'node-schema-panel': nodeSchemaPanelComp,
      'data-schema-panel': dataSchemaPanelComp,
      'preprocess-list-panel': preprocessorListComp,
      'schema-preview-panel': schemaPreviewPanelComp,
    },
  }; //}}}1

  let brokerManager = { //{{{1
    template: '#broker_manager_template',
    inject: ['getGConfig'],
    data: function() {
      return {
        brokerTypeList: brokerTypeList,
        activeBroker: null,
        brokerList: this.getGConfig().brokerList,
        newBroker: {
          name: '',
          type: 'mqtt',
          config: {},
          error: '',
        },
        nameError: '',
        sortField: null,
        sortAscending: true,
      };
    },
    created: function() {
      if (this.brokerList.length > 0)
        this.activeBroker = this.brokerList[0];
    },
    watch: {
      brokerList: function(val) {
        if (this.brokerList.length == 0)
          return;

        if (_.isNull(this.activeBroker))
          this.activeBroker = _.first(this.brokerList);
        else
          this.activeBroker = _.last(this.brokerList);
      },
    },
    methods: {
      addBroker: function() {
        this.newBroker.name = uniqueId('NewBroker', _.map(this.brokerList, function(broker) { return broker.name; }));
        this.newBroker.uniqueKey = uniqueId("BrokerKey", _.map(this.brokerList, function(broker) { return broker.uniqueKey; }));
        this.newBroker.error = '';
        this.newBroker.config = {};
        $(this.$el).find(".modal").modal('show');
        //FIXME: only work for the first time 
        $(this.$el).find("#inputBrokerName").focus();
      },
      createBroker: function() {
        if (!this.isNewNameValid(this.newBroker.name)) {
          return;
        }
        
        // init broker's config
        // common init 
        this.newBroker.config['mqtt_host'] = '';
        this.newBroker.config['mqtt_port'] = 8883;
        this.newBroker.config['mqtt_user'] = '';
        this.newBroker.config['mqtt_passwd'] = '';
        this.newBroker.config['mqtt_client_id'] = '';
        this.newBroker.config['mqtt_qos'] = '1';
        this.newBroker.config['keepalive'] = 60;
        
        //data publish interval
        this.newBroker.config['events_interval'] = 30;

        this.newBroker.config['ca_file'] = '';
        this.newBroker.config['cert_file'] = '';
        this.newBroker.config['key_file'] = '';

        if (this.newBroker.type == "gcp-iot-core") {
          //GCP IoT Core config
          this.newBroker.config['project_id'] = '';
          this.newBroker.config['registry_id'] = '';
          this.newBroker.config['device_id'] = '';
          this.newBroker.config['region'] = 'us-central1';

          //comm config 
          this.newBroker.config['mqtt_host'] = 'mqtt.googleapis.com';
          this.newBroker.config['mqtt_port'] = 8883;

          //tls config
          this.newBroker.config['no_tls'] = false;
        } else if (this.newBroker.type == "aws-iot-core") {
          this.newBroker.config['mqtt_host'] = '';
          this.newBroker.config['mqtt_port'] = 8883;

          //tls config
          this.newBroker.config['no_tls'] = false;
        } else { //for general mqtt broker
          this.newBroker.config['mqtt_host'] = 'test.mosquitto.org';
          this.newBroker.config['mqtt_port'] = 8883;

          //tls config
          this.newBroker.config['no_tls'] = false;
        }

        $(this.$el).find(".modal").modal('hide');
        this.brokerList.push(_.clone(this.newBroker));
      },

      onBrokerUpdated: function(name, config) {
        let activeBroker = this.activeBroker;
        if (!activeBroker)
          return;

        //validate if the new name is unique
        let isNameUnique = _.all(this.brokerList, function(broker) {
          if (broker.uniqueKey == activeBroker.uniqueKey)
            return true;
          return broker.name != name;
        });
        if (!isNameUnique) {
          this.nameError = "The new broker name is not unique";
          return;
        }
        else
          this.nameError = "";

        let oldName = this.activeBroker.name;
        this.$set(this.activeBroker, 'name', name);
        this.$set(this.activeBroker, 'config', _.clone(config));

        if (name != oldName)
          msgbus.$emit("brokerRenamed", oldName, name);
      },

      cloneBroker: function() {
        this.newBroker.name = uniqueId('ClonedBroker', _.map(this.brokerList, function(broker) { return broker.name; }));
        this.newBroker.uniqueKey = uniqueId("BrokerKey", _.map(this.brokerList, function(broker) { return broker.uniqueKey; }));
        this.newBroker.type = this.activeBroker.type;
        this.newBroker.error = '';
        this.newBroker.config = _.clone(this.activeBroker.config);

        //clear device cert and key files
        this.newBroker.config['cert_file'] = '';
        this.newBroker.config['key_file'] = '';

        if (this.newBroker.type == "gcp-iot-core") {
          this.newBroker.config['device_id'] = '';
        } else if (this.newBroker.type == "aws-iot-core") {
        }

        this.brokerList.push(_.clone(this.newBroker));
      },

      deleteBroker: function() {
        if (!confirm("Are you sure to delete this broker(all its topics will be deleted together) ?"))
          return;

        let name = this.activeBroker.name;
        for(let i=0; i<this.brokerList.length; ++i) {
          if (this.brokerList[i].name != this.activeBroker.name)
            continue;

          if (this.brokerList.length > 1)
            this.activeBroker = this.brokerList[(i+1)%this.brokerList.length];
          else
            this.activeBroker = {name: null, type: null};
          this.brokerList.splice(i, 1);
        }
        msgbus.$emit("brokerDeleted", name);

        if (this.brokerList.length == 0)
          this.activeBroker = null;
      },

      isNewNameValid: function(newName) {
        if (!isNameValid(newName)) {
          this.newBroker.error = "Name can only contain alphanumber, '-' and '_', 32 chars at most.";
          return false;
        }

        if (!_.all(this.brokerList, function(broker) {
          return newName != broker.name;
        }))
          this.newBroker.error = "Name already exists";
        else
          this.newBroker.error = '';
        return this.newBroker.error.length == 0;
      },

      sortBy: function(field, ascending) {
        this.sortField = field;
        this.sortAscending = ascending;
        if (ascending)
          this.brokerList.sort(function(a, b) {
            return a[field] > b[field] ? -1 : 1;
          });
        else
          this.brokerList.sort(function(a, b) {
            return a[field] < b[field] ? -1 : 1;
          });
      },

    },
    filters: {
      typeLabel: brokerTypeLabel,
    },
    components: {
      'gcp-iot-core': gcpIoTCoreComp,
      'mqtt': mqttComp,
      'aws-iot-core': awsIoTCoreComp,
    },
  }; //}}}1

  let gcpIoTCoreTopicComp = { //{{{1
    template: '#gcp_iot_core_topic_template',
    props: ['topicName', 'configData', 'nameError', 'broker'],
    inject: ['getGConfig'],
    data: function() {
      return {
        name: this.topicName,
      };
    },
    computed: {
      path: function() {
        let self = this;
        let gConfig = this.getGConfig();
        let brokerConf = _.find(gConfig.brokerList, function(broker) {
          return broker.name == self.broker;
        });
        if (!brokerConf)
          return "";

        return "/devices/" + brokerConf.config['device_id'] + "/events";
      },
      name_error: function() {
        if (this.nameError.length > 0)
          return this.nameError;

        return isNameValid(this.name) ? "" : "Invalid Topic Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
    },
    methods: {
      onSaveLocal: function() {
        this.$emit("topicUpdated", this.name);
      },
      onReset: function() {
        this.$set(this, 'name', this.topicName);
      },
    },
  }; //}}}1

  let mqttTopicComp = { //{{{1
    template: '#mqtt_topic_template',
    props: ['topicName', 'configData', 'nameError'],
    data: function() {
      return {
        name: this.topicName,
        config: _.clone(this.configData),
      };
    },
    mounted: function() {
      if (_.isEmpty(this.config))
        this.$set(this.config, 'path', '');
    },
    computed: {
      name_error: function() {
        if (this.nameError.length > 0)
          return this.nameError;

        return isNameValid(this.name) ? "" : "Invalid Topic Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      path_error: function() {
        return this.config.path.length > 0 ? "" : "Invalid Topic Path";
      },
    },
    methods: {
      onSaveLocal: function() {
        this.$emit("topicUpdated", this.name, this.config);
      },
      onReset: function() {
        this.$set(this, 'name', this.topicName);
        this.$set(this, 'config', _.clone(this.configData));
      },
    },
  }; //}}}1

  let awsIoTCoreTopicComp = { //{{{1
    template: '#aws_iot_core_topic_template',
    props: ['topicName', 'configData', 'nameError'],
    data: function() {
      return {
        name: this.topicName,
        config: _.clone(this.configData),
      };
    },
    mounted: function() {
      if (_.isEmpty(this.config))
        this.$set(this.config, 'path', '');
    },
    computed: {
      name_error: function() {
        if (this.nameError.length > 0)
          return this.nameError;

        return isNameValid(this.name) ? "" : "Invalid Topic Name(only alphanumber, '-' and '_', 32 chars at most)";
      },
      path_error: function() {
        return this.config.path.length > 0 ? "" : "Invalid Topic Path";
      },
    },
    methods: {
      onSaveLocal: function() {
        this.$emit("topicUpdated", this.name, this.config);
      },
      onReset: function() {
        this.$set(this, 'name', this.topicName);
        this.$set(this, 'config', _.clone(this.configData));
      },
    },
  }; //}}}1

  let topicManager = { //{{{1
    template: '#topic_manager_template',
    inject: ['getGConfig'],
    data: function() {
      return {
        topicList: this.getGConfig().topicList,
        brokerList: this.getGConfig().brokerList,
        activeTopic: null,
        newTopic: {
          name: 'NewTopic',
          broker: null,
          config: {},
          error: '',
        },
        nameError: '',
        sortField: null,
        sortAscending: true,
      };
    },
    created: function() {
      if (this.topicList.length > 0)
        this.activeTopic = this.topicList[0];
    },
    mounted: function() {
      msgbus.$on("brokerDeleted", _.bind(this.onBrokerDeleted, this));
      msgbus.$on("brokerRenamed", _.bind(this.onBrokerRenamed, this));
    },
    watch: {
      topicList: function(val) {
        if (this.topicList.length == 0)
          return;

        if (this.activeTopic == null)
          this.activeTopic = _.first(this.topicList);
        else
          this.activeTopic = _.last(this.topicList);
      },
    },
    computed: {
      activeTopicType: function() {
        return this.topicTypeName(this.activeTopic.broker);
      },
    },
    methods: {
      isNewNameValid: function(newName) {
        if (!isNameValid(newName)) {
          this.newTopic.error = "Name can only contain alphanumber, '-' and '_', 32 chars at most.";
          return false;
        }

        if (!this.validateGCPIotCoreTopics())
          return false;

        //check if topic is unique 
        if (!_.all(this.topicList, function(topic) {
          return newName != topic.name;
        }))
          this.newTopic.error = "Name already exists";
        else
          this.newTopic.error = '';
        return this.newTopic.error.length == 0;
      },

      validateGCPIotCoreTopics: function() {
        let self = this;
        let brokerNode = this.brokerNode(this.newTopic.broker);
        if (brokerNode && brokerNode.type == "gcp-iot-core") {
          if (_.any(this.topicList, function(topic) {
            return topic.broker == self.newTopic.broker;
          })) {
            // this.$set(this.newTopic, 'error', "GCP IoT Core type broker can only include one topic.");
            this.newTopic.error = "GCP IoT Core type broker can only include one topic.";
            return false;
          }
        }
        return true;
      },
      addTopic: function() {
        this.newTopic.name = uniqueId('NewTopic', _.map(this.topicList, function(topic) { return topic.name; }));
        this.newTopic['uniqueKey'] = uniqueId('TopicKey', _.map(this.topicList, function(topic) { return topic.uniqueKey; }));
        if (!this.newTopic.broker) {
          if (this.brokerList.length > 0)
            this.newTopic.broker = this.brokerList[0].name;
        } else {
          let prevBroker = this.newTopic.broker;
          let isPrevBrokerValid = _.any(this.brokerList, function(broker) {
            return broker.name == prevBroker;
          });
          if (!isPrevBrokerValid)
            this.newTopic.broker = null;
        }

        this.newTopic.error = '';
        this.newTopic.config = {};
        $(this.$el).find(".modal").modal('show');
        //FIXME: only work for the first time 
        $(this.$el).find("#inputNewTopicName").focus();
      },
      createTopic: function() {
        if (!this.isNewNameValid(this.newTopic.name) || !this.newTopic.broker) {
          return;
        }

        let brokerNode = this.brokerNode(this.newTopic.broker);
        if (brokerNode.type == 'mqtt' || brokerNode.type == 'aws-iot-core')
          this.newTopic.config['path'] = '';

        $(this.$el).find(".modal").modal('hide');
        this.topicList.push(_.clone(this.newTopic));
      },
      deleteTopic: function() {
        if (!confirm("Are you sure to delete this topic ?"))
          return;

        for(let i=0; i<this.topicList.length; ++i) {
          if (this.topicList[i].name != this.activeTopic.name)
            continue;

          if (this.topicList.length > 1)
            this.activeTopic = this.topicList[(i+1)%this.topicList.length];
          else
            this.activeTopic = {name: null, broker: null};
          this.doDeleteTopic(i);
          break;
        }
      },

      doDeleteTopic: function(index) {
        if (index >= this.topicList.length)
          return;

        let name = this.topicList[index].name;
        this.topicList.splice(index, 1);
        msgbus.$emit("topicDeleted", name);
      },

      sortBy: function(field, ascending) {
        this.sortField = field;
        this.sortAscending = ascending;
        if (ascending)
          this.topicList.sort(function(a, b) {
            return a[field] > b[field] ? -1 : 1;
          });
        else
          this.topicList.sort(function(a, b) {
            return a[field] < b[field] ? -1 : 1;
          });
      },

      onTopicUpdated: function(name, config) {
        let activeTopic = this.activeTopic;
        if (!activeTopic)
          return;

        let isNameUnique = _.all(this.topicList, function(topic) {
          if (topic.uniqueKey == activeTopic.uniqueKey)
            return true;
          return topic.name != name;
        });
        if (!isNameUnique) {
          this.nameError = "The new topic name is not unique";
          return;
        }
        else
          this.nameError = "";

        let oldName = this.activeTopic.name;
        this.$set(this.activeTopic, 'name', name);
        if (config)
          this.$set(this.activeTopic, 'config', _.clone(config));

        if (oldName != name)
          msgbus.$emit("topicRenamed", oldName, name);
      },

      onBrokerDeleted: function(name) {
        for(let i=this.topicList.length; i>0; --i) {
          let index = i-1;
          let topic = this.topicList[index];
          if (topic.broker != name)
            continue;

          this.doDeleteTopic(index);
        }
      },
      onBrokerRenamed: function(oldName, newName) {
        for(let i=this.topicList.length; i>0; --i) {
          let index = i-1;
          let topic = this.topicList[index];
          if (topic.broker != oldName)
            continue;

          topic.broker = newName;
        }
      },

      brokerNode: function(brokerName) {
        return _.find(this.brokerList, function(broker) {
          return broker.name == brokerName;
        });
      },

      topicTypeName: function(brokerName) {
        let brokerType = 'mqtt';
        _.any(this.brokerList, function(broker) {
          if (broker.name == brokerName) {
            brokerType = broker.type;
            return true;
          } else
            return false;
        });
        return brokerType+"-topic";
      },
    },
    components: {
      'gcp-iot-core-topic': gcpIoTCoreTopicComp,
      'mqtt-topic': mqttTopicComp,
      'aws-iot-core-topic': awsIoTCoreTopicComp,
    },
  }; //}}}1

  //the global config for the whole DataService. the config data structure is
  //flat in js side, but it will be merge into a tree structure before sending
  //to controller; when loading the config from controller, we need to
  //reorganize it into the flat structure for js to use
  let gConfig = { //{{{1
    brokerList: [],
    topicList: [],
  }; //}}}1

  //create&init root object
  window.app = new Vue({ // {{{1
    el: '#data_service_config_content',
    data: {
      sections: [
        {sec_id: 'broker-manager', name: 'Broker Manager'},
        {sec_id: 'topic-manager', name: 'Topic Manager'},
        {sec_id: 'data-mapping', name: 'Message Manager'},
      ],
      active_sec_id: 'broker-manager',
      spinner: null,
    },
    mounted: function() {
      this.loadConfig();
    },
    methods: {
      isSectionActive: function(sec) { return sec.sec_id == this.active_sec_id; },
      getSpinner() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },
      getGConfig: function() { return gConfig; },

      applyDataMappingData: function(resp) { 
        if (!_.has(resp, 'brokerList')) {
          console.debug("can not find brokerList, skip");
          return;
        }

        let config = this.getGConfig();
        _.each(resp.brokerList, function(broker, brokerIndex) {
          let brokerUniqueKey = "BrokerKey_" + brokerIndex;
          var brokerData = {uniqueKey: brokerUniqueKey, name: broker.name, type: broker.type, config: _.clone(broker.config)};
          config.brokerList.push(brokerData);
          
          if (_.has(broker, 'topicList') && _.isArray(broker.topicList)) {
            _.each(broker.topicList, function(topic, topicIndex) {
              let topicUniqueKey = "TopicKey_"+brokerIndex + "_" + topicIndex;
              var topicData = {uniqueKey: topicUniqueKey, name: topic.name, broker: broker.name, config: _.clone(topic.config)};
              config.topicList.push(topicData);
            });
          }
        });
      },
      applyDataMappingConfig: function(resp) {
        if (!_.has(this.$refs, 'data-mapping'))
          return;

        let dataMappingInst = this.$refs['data-mapping'][0];
        dataMappingInst.initData(resp);
      },
      loadConfig: function() {
        let self = this;
        $.ajax({
          url: 'index.php',
          method: 'GET',
          dataType: 'json',
          data: {
            'action': 'load_config'
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data, status, jq) {
            if (data.redirect)
              redirect(data.redirect);
            
            var resp_data = _.has(data, 'data') ? data.data : null;
            if (resp_data) {
              if (_.has(resp_data, 'certs_dir')) {
                let config = self.getGConfig();
                self.$set(config, 'certs_dir', resp_data.certs_dir);
              }
              if (_.has(resp_data, 'config_data')) {
                self.applyDataMappingConfig(JSON.parse(resp_data.config_data));
              }

              if (_.has(resp_data, 'mapping_data')) {
                self.applyDataMappingData(JSON.parse(resp_data.mapping_data));
              }
            }
          },
          error: function(data, status) {
            alert("Failed to load config data: " + data); 
            // msgbus.$emit("alert", "Error", "Failed to load config data: " + data, 'error'); 
          },
          complete: function() {
            self.getSpinner().stop();
          },
        });
      },

      prepareMappingData: function() { 
        let gConfig = this.getGConfig();
        let config = {
          'version': 1, 
          'timestamp': (new Date()).getTime(),
          'certs_dir': gConfig.certs_dir,
          'brokerList': []
        };

        let mappingList = null;
        if (_.has(this.$refs, 'data-mapping'))  {
          let dataMappingInst = this.$refs['data-mapping'][0];
          mappingList = dataMappingInst.getMappingData();
        }
        
        _.each(gConfig.brokerList, function(broker) {
          let brokerData = {
            name: broker.name, 
            type: broker.type, 
            config: _.clone(broker.config)
          };
          config['brokerList'].push(brokerData);

          if (_.has(gConfig, 'topicList')) {
            brokerData['topicList'] = [];
            _.each(gConfig.topicList, function(topic) {
              if (topic.broker != brokerData.name)
                return;

              let topicData = {
                name: topic.name, 
                broker: topic.broker, 
                config: _.clone(topic.config)
              };

              if (!_.isNull(mappingList)) {
                _.each(mappingList, function(mapping) {
                  if (mapping.topic != topic.name)
                    return;
                  topicData['dataMapping'] = mapping;
                  return;
                });
              }
              brokerData['topicList'].push(topicData);
            });
          }
        });

        return config; 
      },
      prepareJsTreeConfig: function() {
        let configData = {};
        if (_.has(this.$refs, 'data-mapping'))  {
          let dataMappingInst = this.$refs['data-mapping'][0];
          configData = dataMappingInst.getConfigData();
        }
        configData['version'] = 1;
        configData['timestamp'] = (new Date()).getTime();
        return configData;
      },
      isInputsValid: function() {
        let gConfig = this.getGConfig();

        let allBrokerValid = _.all(gConfig.brokerList, function(broker) {
          if (!_.has(broker, 'errors'))
            return true;
          return _.all(_.keys(broker.errors), function(key) {
            let has_error = _.has(broker.errors, key) && broker.errors[key] == true;
            return !has_error;
          });
        });
        if (!allBrokerValid)
          return false;

        let allTopicValid = _.all(gConfig.topicList, function(topic) {
          if (!_.has(topic, 'errors'))
            return true;
          return _.all(_.keys(topic.errors), function(key) {
            let has_error = _.has(topic.errors, key) && topic.errors[key] == true;
            return !has_error;
          });
        });
        return allBrokerValid && allTopicValid;
      },
      saveConfig: function() {
        if (!this.isInputsValid())
        {
          msgbus.$emit("alert", "Error", "Some inputs are invalid, please fix it and try again.", 'error', 0); 
          return;
        }
        else
          msgbus.$emit("alert:clear");
        let mappingData = this.prepareMappingData();

        if (!mappingData) {
          console.warn("mapping data is invalid");
          return;
        }

        let self = this;
        $.ajax({
          url: 'index.php',
          method: 'POST',
          dataType: 'json',
          data: {
            'action': 'save_config',
            'mapping_data': JSON.stringify(mappingData),
            'config_data': JSON.stringify(this.prepareJsTreeConfig()),
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data, status, jq) {
            if (data.redirect)
              redirect(data.redirect);
          },
          error: function(data, status) {
            alert("Failed to save config data: " + data); 
            // msgbus.$emit("alert", "Error", "Failed to load config data: " + data, 'error'); 
          },
          complete: function() {
            self.getSpinner().stop();
          },
        });
      },
    },
    watch: {
      active_sec_id: function(val) {
        msgbus.$emit('alert:clear');
      },
    },
    provide: function() {
      return {
        getGConfig: this.getGConfig,
        getTopicList: function() {
          if (!_.has(gConfig, 'topicList'))
            gConfig['topicList'] = [];
          return gConfig.topicList;
        },
      };
    },
    components: {
      'broker-manager': brokerManager,
      'topic-manager': topicManager,
      'data-mapping': dataMappingComp,
      'alert': alertComp,
      'file-selector': fileSelectorComp,
    }
  }); //}}}1

});

