(function () {
  var L = function(str) { return l18n(str, 'cpt.datasources.js');};
  var isBlank = function(str) { return String(str).replace(/^\s+|\s+$/g, '').length == 0; };

  var cptWebServiceDatasource = function (settings, updateCallback) {
    var self = this;
    var maxUrlLength = 1000;
    var updateTimer = null;
    var currentSettings = settings;
    var reqUrlPayLoads = [];

    var authViewModel = {
      username: ko.observable(''),
      password: ko.observable(''),
      error_msg: ko.observable('')
    } 
    
    //set breakpoint whenever 'name' property changed
    // currentSettings._name = currentSettings.name;
    // Object.defineProperty(currentSettings, "name", {
    //   get: function() {
    //     return currentSettings._name;
    //   },
    //   set: function(value) {
    //     debugger;
    //     currentSettings._name = value;
    //   }
    // });

    function controllerHostLabel() {
      if (!currentSettings.controllerHost || currentSettings.controllerHost.length === 0)
        return window.location.host
      else
        return currentSettings.controllerHost
    }
    
    function commonPath(pathList) {
      if (pathList.length == 1) {
        var parts = pathList[0].split("/");
        parts.pop();
        return parts.join("/");
      }
      var commonParts = _.reduce(pathList, function(memo, path) {
        var parts = path.split("/");
        if (_.isEmpty(memo))
          return parts;
        else {
          for(i=0; i<memo.length; ++i) {
            if (parts.length <= i)
              return parts;

            if (memo[i] != parts[i])
              return _.first(memo, i)
          }
          return memo;
        }
      }, []);
      return commonParts.join("/");
    }
    
    function parseValue(item) {
      var type = item['type'];
      var value = item['value'];
      var slotType = item['slotType'];
      if (type == 'double' || type == 'float')
        return parseFloat(value);
      else if (type == 'byte' || type == 'short' || type == 'int' || type == 'long')
        return parseInt(value);
      else if (type == 'bool') {
        if (value == 'true')
          return true;
        else if (value == 'false')
          return false;
        else
          return null;
      }
      else
        return value;
    }

    function processSlotData(slotDatas) {
      return _.object(
          _.map(slotDatas, function(item) { return item['name']; }), 
          _.map(slotDatas, function(item) { return parseValue(item); }));
    }

    function processCompData(compDatas) {
      _.each(compDatas, function(compData, index) { 
        if (_.isArray(compData)) {
          var objData = processCompData(compData);
          objData['path'] = commonPath(_.keys(objData));
          compDatas[index] = objData;
        }
        else {
          compData = _.extend(compData, processSlotData(compData.slots));
          delete compData['slots'];
        }
      });

      //convert component data list into hash indexed by 'path'
      return _.object(_.map(compDatas, function(compData) {
        var path = compData['path'];
        delete compData['path'];
        return path;
      }), compDatas);
    }

    function parseCompTypeData(dsName, compListData) {
      return _.reduce(compListData, function(memo, compDatas) {
        return _.extend(memo, 
                         _.object(_.pluck(compDatas, 'path'), 
                                  _.map(_.pluck(compDatas, 'slots'),
                                        function(slots) { 
                                          return _.object(_.pluck(slots, 'name'), slots); 
                                        }
                                  ))
                         ); 
      }, window.g_compTypeData[dsName]);
    }
    
    function buildCompTypeData(data) {
      if (_.isUndefined(window.g_compTypeData))
        window.g_compTypeData = {};
      
      var name = currentSettings.name;
      if (_.isUndefined(window.g_compTypeData[name]))
        window.g_compTypeData[name] = {};

      data = _.flatten(data);
      if (_.every(data, _.isArray))
        parseCompTypeData(name, data);
      else
        parseCompTypeData(name, [data]);
    }

    function updateRefresh() {
      if (updateTimer) {
        clearTimeout(updateTimer);
        updateTimer = null;
      }
      self.updateNow();
    }
    
    function buildReqUrlPayLoads() {
      if (reqUrlPayLoads.length > 0)
        return;

      var urls = _.pluck(currentSettings.compPathList, 'compPath');
      if (_.isEmpty(urls))
        return;
      
      for(var i=0; i<urls.length; ++i) {
        var url = urls[i];
        if (reqUrlPayLoads.length == 0) {
          reqUrlPayLoads.push(url);
          continue;
        }

        var oneBatch = _.last(reqUrlPayLoads);
        if ((oneBatch.length + url.length + 1) <= maxUrlLength) {
          oneBatch += '~' + url;
          reqUrlPayLoads[reqUrlPayLoads.length-1] = oneBatch;
        } else 
          reqUrlPayLoads.push(url);
      }
      return reqUrlPayLoads;
    }

    this.scheduleNextUpdate = function() {
      if (updateTimer) {
        clearTimeout(updateTimer);
        updateTimer = null;
      }

      var refreshTime = (reqUrlPayLoads.length > 0) ? 1*1000 : currentSettings.refresh * 1000;
      updateTimer = setTimeout(self.updateNow, refreshTime);
    }
    
    this.updateNow = _.debounce(function () {
      if (reqUrlPayLoads.length == 0)
        buildReqUrlPayLoads();

      if (reqUrlPayLoads.length == 0)
        return;
      
      var payload = reqUrlPayLoads.shift(); 

      var protocol = currentSettings.useHTTPS ? "https" : "http";
      var requestURL = protocol + '://' + controllerHostLabel() + '/sdcard/cpt/app/data_api.php?url=/app/objects' + payload;
      
      $.ajax({
        url: requestURL,
        dataType: "json",
        type: "GET",
        xhrFields: { withCredentials: true },
        headers: {'X-Requested-With': 'XMLHttpRequest'},
        success: function (data) {
          if (data.response) {
            buildCompTypeData(data.response.data);

            updateCallback(processCompData(data.response.data));

            self.scheduleNextUpdate();
          } else {
            console.warn(data);
            if (data.redirect && /signin\.php$/.test(data.redirect))
              showAuthDialog();
            else 
              self.scheduleNextUpdate();
          }
        },
        error: function (xhr, status, error) {
          console.warn("failed to get data from " + requestURL + " status: " + status + "error: " + error);
          self.scheduleNextUpdate();
        }
      });
    }, 200, true);

    this.onDispose = function () {
      clearTimeout(updateTimer);
      updateTimer = null;
    }

    this.onSettingsChanged = function (newSettings) {
      currentSettings = _.defaults(newSettings, currentSettings);
      updateRefresh();
    }
   
    function showAuthDialog() {
        if (!isBlank(currentSettings.userName))
          authViewModel.username(currentSettings.userName);
        else 
          authViewModel.username("admin");
      
        if (!isBlank(currentSettings.password))
          authViewModel.password(currentSettings.password);
      
        if (isBlank(authViewModel.username()) || isBlank(authViewModel.password()))
        {
          var dom = $("<div><div style='color:red; margin-top:5px; margin-bottom:5px;' data-bind='html: error_msg'></div><div style='margin-bottom:15px;'><div style='width:80px; float:left;'><label for='user_name'>UserName:</label></div><input type='text' id='user_name' data-bind='value: username' autofocus></div><div><div style='width:80px; float:left;'><label for='password'>Password:</label></div><input type='password' id='password' data-bind='value: password'></div></div>");
          ko.applyBindings(authViewModel, dom[0]);
          freeboard.showDialog(dom, L("CPT Web Service Authentication({host})").supplant({host: controllerHostLabel()}),
              L("Ok"), L("Cancel"), loginCptWebService);
        }
        else
          loginCptWebService();
    }
    
    function loginCptWebService() {
      authViewModel.error_msg('');
      var username = authViewModel.username();
      var password = authViewModel.password();
      if (!username || !password) {
        authViewModel.error_msg(L("invalid username or password"));
        return true;
      }

      freeboard.showLoadingIndicator(true);
      var protocol = currentSettings.useHTTPS ? "https" : "http";
      var url = protocol + '://' + controllerHostLabel() + '/sdcard/cpt/app/signin.php';
      $.ajax({
        url: url, type: "GET", dataType: 'json',
        xhrFields: { withCredentials: true },
        headers: {'X-Requested-With': 'XMLHttpRequest'},
        data: {'user[name]': username}
      }).done(function(data) { 
        if (!data.authToken) {
          console.error("failed to fetch authToken");
          freeboard.showLoadingIndicator(false);
          return; 
        }
        var sections = data.authToken.split('_'); if (sections.length != 2) {
          console.warn("invalid authToken: " + data.authToken);
          freeboard.showLoadingIndicator(false);
          return; 
        }
        var token1 = sections[0]; var token2 = sections[1];
        // to calculate password hash, there are 2 steps:
        // 1. calculate SHA-1 hash for password + token1
        // 2. calculate SHA-1 hash for result of step1 + token2
        var shaObj1 = new jsSHA(password + token1, 'TEXT');
        var shaObj2 = new jsSHA(shaObj1.getHash('SHA-1', 'HEX') + token2,
            'TEXT');
        var authHash = shaObj2.getHash('SHA-1', 'HEX');
        $.ajax({
          url: url, type: 'POST', dataType: 'json',
          xhrFields: { withCredentials: true },
          headers: {'X-Requested-With': 'XMLHttpRequest'},
          data: {
            'user[name]': username, 'user[authHash]': authHash
          }
        }).done(function(data) {
          freeboard.showLoadingIndicator(false);
          if (data.error) {
            authViewModel.error_msg(data.error.text);
            showAuthDialog();
            return;
          } else {
            console.debug("signin done");
            _.each(_.uniq(window.cptWebServiceLoginCallbacks[currentSettings.controllerHost]), function(callback) { callback(); });
          }
        });
      });
      return false;
    } 
    
    if (!window.cptWebServiceLoginFuncs)
      window.cptWebServiceLoginFuncs = {};
    if (!window.cptWebServiceLoginFuncs[currentSettings.controllerHost]) {
      //NOTE: if data source host is the current host, then authentication is not required 
      if (currentSettings.controllerHost.length !== 0 && currentSettings.controllerHost != window.location.host) {
        var showDlg = _.once(showAuthDialog)
        window.cptWebServiceLoginFuncs[currentSettings.controllerHost] = function(callback) {
          if (!window.cptWebServiceLoginCallbacks)
            window.cptWebServiceLoginCallbacks = {};
          if (!window.cptWebServiceLoginCallbacks[currentSettings.controllerHost])
            window.cptWebServiceLoginCallbacks[currentSettings.controllerHost] = [];
          window.cptWebServiceLoginCallbacks[currentSettings.controllerHost].push(callback);
          showDlg();
        }
      }
    }

    if (window.cptWebServiceLoginFuncs[currentSettings.controllerHost]) 
      window.cptWebServiceLoginFuncs[currentSettings.controllerHost](updateRefresh);
    else
      updateRefresh();
  };

  freeboard.loadDatasourcePlugin({
    type_name: "CPTWebService",
    display_name: L("CPT Web Service"),
    description: L("interface to access CPT Web Service data"),
    external_scripts : ["plugins/cpt/sha1.js"], 
    settings: [
      {
        name: "controllerHost",
        display_name: L("Controller Host"),
        type: "text",
        default_value: "",
        description: L("for example: 192.168.10.11, 192.168.10.11:8080. leave blank will use current host.")
      },
      {
        name: "userName",
        display_name: L("UserName"),
        type: "text",
        default_value: "admin",
        description: L("username to login controller.")
      },
      {
        name: "useHTTPS",
        display_name: L("Use HTTPS"),
        type: "boolean", 
        default_value: '',
        description: L("use HTTPS protocol to communicate with this controller")
      },
      {
        name: "password",
        display_name: L("Password"),
        type: "password",
        default_value: "",
        description: L("password for above user")
      },
      {
        name: "compPathList",
        display_name: L("Component Path List"),
        type: "array",
        description: L("for example: /Folder/Ramp ; /Folder/* (all data under '/Folder')"),
        settings: [
          {
            name: 'compPath',
            display_name: L('Component Path'),
            type: 'text'
          }
        ]
      },
      {
        name: "refresh",
        display_name: L("Refresh Every"),
        type: "number",
        suffix: L("seconds"),
        default_value: 5
      }
    ],
    newInstance: function (settings, newInstanceCallback, updateCallback) {
      newInstanceCallback(new cptWebServiceDatasource(settings, updateCallback));
    }
  });
}());
// vim: ts=2 sw=2 softtabstop=1
