$(function() {

  let dateTimeComp = {
    template: '#datetime-panel-template',
    fp: null,
    data: function() {
      return {
        curDateTime: null,
        curTZ: '',
        tz_list: [],
        utcOffset: 0,
        utcOffsetMode: 'OS',

        error: null,
        spinner: null,
      };
    },
    computed: {
      utcOffsetStr: function() {
        return Number(this.utcOffset/3600.0).toString() + " hours";
      },
    },
    watch: {
      curTZ: function(newVal, oldVal) {
        if (!this.curDateTime)
          return;

        this.curDateTime.setTimezone(newVal);
        if (this.fp)
          this.fp.setDate(this.curDateTime.toString('yyyy-MM-dd HH:mm:ss'));

        this.utcOffset = -1*this.curDateTime.getTimezoneOffset()*60;
      }
    },
    mounted: function() {
      this.getSpinner().spin(this.$el);
      this.loadConfig();
    },
    methods: {
      getSpinner: function() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },
      sysclockInNs: function() {
        return (this.curDateTime.getTime() - Date.UTC(2000, 0, 1, 0, 0, 0)) + "000000";
      },
      newDateTime: function() {
        if (this.fp) {
          let newDT = flatpickr.formatDate(_.first(this.fp.selectedDates), "Y-m-d H:i:S");
          return new timezoneJS.Date(newDT, this.curTZ);
        }
        else
          return null;
      },
      loadConfig: function() {
        let self = this;
        $.ajax({
          type: 'GET',
          url: "../../app/data_api.php?url=/app/objects/service/time",
          dataType: 'json',
          success: function(data) {
            if (data.redirect)
              window.location.href = data.redirect
            else if (data.error)
              self.error = data.error.text;
            else
              self.handleResponse(data.response);
          },
          complete: function() {
            self.onConfigLoaded();
          },
        });
        if (timezoneJS.timezone.getAllZones().length == 0) {
          timezoneJS.timezone.loadingScheme = timezoneJS.timezone.loadingSchemes.PRELOAD_ALL;
          timezoneJS.timezone.zoneFileBasePath = '../../app/tz';
          timezoneJS.timezone.init({callback: _.bind(this.onTZDataLoaded, this)})
        }
        else
          self.onConfigLoaded();
      },
      handleResponse: function(resp) {
        if (_.has(resp, "resultCode") && resp.resultCode != 0) {
          this.error = "Can not access data time object, please make sure the datetime object's path(/service/time) is configured correctly";
          return;
        }
        if (!_.isArray(resp.data) || resp.data.length != 1)
        {
          this.error = "Invalid response data";
          return;
        }

        let self = this;
        let tz = null;
        _.each(resp.data[0].slots, function(slot) {
          if (slot.name == 'utcOffset')
            self.utcOffset = parseInt(slot.value);
          else if (slot.name == 'osUtcOffset')
            self.utcOffsetMode = slot.value == "true" ? "OS" : "Configuration";
          else if (slot.name == 'tz') 
            tz = slot.value;
          else if (slot.name == 'nanos') {
            let nanos = slot.value;
            let secs = nanos/1000000000 + Date.UTC(2000, 0, 1, 0, 0, 0)/1000;
            self.curDateTime = new timezoneJS.Date(secs*1000, 'UTC')
          }
        });
        this.curTZ = tz;
      },
      onTZDataLoaded: function() {
        this.tz_list = timezoneJS.timezone.getAllZones();
        this.onConfigLoaded();
      },
      onConfigLoaded: _.after(2, function() {
        this.getSpinner().stop();

        this.fp = $(this.$el).find("#inputDateTime").flatpickr({
          enableTime: true,
          defaultDate: this.curDateTime.toString('yyyy-MM-dd HH:mm:ss'),
          dateFormat: "Y-m-d H:i:S",
          time_24hr: true,
        });
      }),

      postRequest: function(path, value, slotType, dataType, opts) {
        let self = this;
        $.ajax({
          type: "POST",
          url: "../../app/data_api.php?url=/app/objects/service/time",
          timeout: 30000,
          data: {
            path: path,
            type: dataType,
            value: value,
            slotType: slotType
          },
          dataType: 'json',
          success: function(data) {
            if (data.redirect)
              window.location.href = data.redirect
            else if (data.error)
              self.error = data.error.text;
            else
              if (opts && _.isFunction(opts.success))
                opts.success();
          },
          error: function(data, status) {
            self.error = "Error: " + status;
            if (opts && _.isFunction(opts.error))
              opts.error();
          },
          complete: function() {
            if (opts && _.isFunction(opts.complete))
              opts.complete();
          }
        })
      },
      saveConfig: function() {
        let self = this;
        this.error = null;
        this.curDateTime = this.newDateTime();

        let path = "/app/objects/service/time";
        let isOSUTCOffsetMode = this.utcOffsetMode == "OS";
        this.getSpinner().spin(this.$el);
        this.postRequest(path+'.osUtcOffset', isOSUTCOffsetMode, 'property', 'bool', {
          success: function() {
            self.postRequest(path+'.tz', self.curTZ, "property", 'Buf', {
              success: function() {
                if (isOSUTCOffsetMode)
                  self.postRequest(path+'.setSysClock', self.sysclockInNs(), 'action', 'long', {
                    complete: function() { self.getSpinner().stop(); }
                  });
                else
                  self.postRequest(path+'.utcOffset', self.utcOffset, 'property', 'int', {
                    success: function() {
                      self.postRequest(path+'.setSysClock', self.sysclockInNs(), 'action', 'long', {
                        complete: function() { self.getSpinner().stop(); }
                      });
                    },
                    error: function() {
                      self.getSpinner().stop();
                    },
                  });
              },
              error: function() {
                self.getSpinner().stop();
              },
            });
          },
          error: function() {
            self.getSpinner().stop();
          },
        })
      },
    },
  };

  let ntpComp = {
    template: '#ntp-panel-template',
    data: function() {
      return {
        timeServer: 'time.windows.com',
        timeServerError: false,

        updateInterval: 1,
        updateIntervalError: false,

        enableTimeSync: '1',

        error: null,

        spinner: null,
      };
    },
    mounted: function() {
      this.loadConfig();
    },
    methods: {
      getSpinner: function() {
        if (!this.spinner)
          this.spinner = new Spinner();
        return this.spinner;
      },
      loadConfig: function() {
        let self = this;
        $.ajax({
          type: 'GET',
          url: 'datetime_config.php',
          data: {
            action: 'getTimeServer',
          },
          dataType: 'json',
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data, status, xhr) {
            if (data.redirect)
              window.location.href = data.redirect
            else if (data.error)
              self.error = data.error.text;
            else {
              if (_.has(data, 'config')) {
                let config = data.config;
                if (_.has(config, 'timeServer'))
                  self.timeServer = config.timeServer;
                if (_.has(config, 'updateInterval'))
                  self.updateInterval = parseInt(config.updateInterval);
                if (_.has(config, 'enableTimeSync'))
                  self.enableTimeSync = config.enableTimeSync;
              }
            }
          },
          error: function(xhr, status, error) {
            self.error = error;
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },
      validateInputs: function() {
        this.timeServerError = !/^[a-zA-Z0-9][-.a-zA-Z0-9]*[a-zA-Z0-9]$/.test(this.timeServer);
        this.updateIntervalError = this.updateInterval <= 0;
        return !this.timeServerError && !this.updateIntervalError;
      },
      saveConfig: function() {
        if (!this.validateInputs())
          return;

        let self = this;
        $.ajax({
          type: 'POST',
          url: "datetime_config.php",
          dataType: 'json',
          data: {
            action: 'updateTimeServer',
            enableTimeSync: this.enableTimeSync,
            timeServer: this.timeServer,
            updateInterval: this.updateInterval,
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data, status, xhr) {
            if (data.redirect)
              window.location.href = data.redirect
            else if (data.error)
              self.error = data.error.text;
            else {
            }
          },
          error: function(xhr, status, error) {
            self.error = error;
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },
      
      syncDateTime: function() {
        if (this.enableTimeSync != '1')
          return;

        let self = this;
        $.ajax({
          type: 'POST',
          url: "datetime_config.php",
          dataType: 'json',
          data: {
            action: 'syncDateTime',
          },
          beforeSend: function() {
            self.getSpinner().spin(self.$el);
          },
          success: function(data, status, xhr) {
            if (data.redirect)
              window.location.href = data.redirect
            else if (data.error)
              self.error = data.error.text;
            else {}
          },
          error: function(xhr, status, error) {
            self.error = error;
          },
          complete: function() {
            self.getSpinner().stop();
          }
        });
      },

    },
  };

  let app = new Vue({ //{{{1
    el: '#datetime_config_content',
    data: {
      sections: [
        {sec_id: 'datetime-panel', name: 'DateTime'},
        {sec_id: 'ntp-panel', name: 'NTP'},
      ],
      active_sec_id: 'datetime-panel',
    },
    methods: {
      isSectionActive: function(sec) { return sec.sec_id == this.active_sec_id; },
    },
    components: {
      'datetime-panel': dateTimeComp,
      'ntp-panel': ntpComp,
    },
  });
});

// vim: foldmethod=marker ts=2 sw=2
