jQuery ->

  ##########################################################
  # Extend jQuery
  ##########################################################
  $.fn.textWidth = (text, font)->
    if (!$.fn.textWidth.fakeEl)
      $.fn.textWidth.fakeEl = $('<span>').hide().appendTo(document.body)
    $.fn.textWidth.fakeEl.html(text || this.val() || this.text()).css('font', font || this.css('font'))
    $.fn.textWidth.fakeEl.width()

  ##########################################################
  # Helper Functions
  ##########################################################
  attrMapListToMap = (attributes)->
    return {} unless attributes?
    _.reduce(attributes, (memo, attr)=>
            memo[attr.name] = attr.value
            memo
    {})

  boolToNumber = (bool)->
    switch bool
      when 'false'
        0
      when 'true'
        1
      when 'null'
        2
      else
        console.error "got invalid bool value: #{bool}"
        0

  getUrlParam = (sVar)->
    param = unescape(
      window.location.search.replace(
        new RegExp("^(?:.*[&\\?]" + escape(sVar).replace(
          /[\.\+\*]/g, "\\$&") + "(?:\\=([^&]*))?)?.*$", "i"),
        "$1")
    )
    if _.str.isBlank(param) then null else param

  childElementsByTagName = (node, tagName...)->
    child for child in node.childNodes when child.nodeType == 1 and child.tagName in tagName

  domAttrsToMap = (node)->
    result = {}
    for index in [0...node.attributes.length]
      attr = node.attributes.item(index)
      result[attr.name] = attr.value
    return result

  compPath2compId = (compPath)->
    # TODO: should remove this method later when all users switch to version 1.0 and later
    grDoc = window.grCanvasView.model
    # convert compPath only when version is missing or <1.0
    if grDoc.version() >= 1.0
      return compPath

    newPath = compPath.split('/')
    newPath.shift()
    newPath.shift()
    '/' + newPath.join('/')

  maxZIndex = ->
    _.max($('div#grCanvas .grObject').map ->
      $(this).css('z-index')
    .get())

  validateNormalText = (input, name='input', minLength=8)->
    if input.length < minLength
      return L("{name} should be longer than {minLength} characters").supplant
        name: name, minLength: minLength

    maxLength = 100
    if input.length >= maxLength
      return L("{name} should be shorter than {maxLength} characters").supplant
        name: name, maxLength: maxLength
        

    if /^[-_@.a-zA-Z0-9]+$/.test(input)
      return null
    else
      return L("{name} must be alphabet, numbers, and some symbols(- _ . @)").supplant
        name: name

  redirect = (url)->
    window.location.href = url

  # form helpers
  renderError = (elem, text)->
    $("<span class='help-inline hide'></span>").insertAfter(elem).text(text).show()
    elem.parent('.controls').parent('.control-group').addClass('error')

  isElemAttached = (elem)->
    elem.closest("html").length > 0

  rgbToHex = (strRGB)->
    [r, g, b, a] = _.map(strRGB.split(","), _.str.toNumber)
    rgb = b | (g << 8) | (r << 16)
    (0x1000000 | rgb).toString(16).substring(1)

  isGraphicLink = (link)->
    _.str.endsWith(link, '.gr') and (not _.str.startsWith(link, 'http'))
    
  logAndReturn = (msg)->
    console.log(msg)
    msg
    
  isUserProp = (propName)->
    _.str.startsWith(propName, '@')
    
  curUserName = ()->
    _.str.trim($("#userBtn > a:first").text())
    
  randStr = (length=16)->
    _.first(_.shuffle("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"), length).join('')
    
  L = (str)->
    l18n(str, "graphic.js")
    
  class TextFormatter extends Backbone.Model
    @applyFontInfo: (object, font)->
      return if !font
      [family, pt, px, styleHint, weight, style, underline, strikeOut, others...] = font.split(",")
      object.css("font-family", family)

      if pt == '-1' or pt == -1
        height = object.css("height")
        if height
          # object.css("line-height", height)
          height = _.str.toNumber(height.slice(0, -2))
          ems = height/16.0
          object.css("font-size", "#{ems}em")
      else
        object.css("font-size", "#{pt}pt")

      object.css("font-style", ["normal", "italic", "oblique"][style])
      object.css("font-weight", @convertWeight(weight))
      object.css("line-height", "100%")

      decoration = []
      if !!underline and underline == '1'
        decoration.push 'underline'
      if !!strikeOut and strikeOut == '1'
        decoration.push "line-through"
      object.css("text-decoration", decoration.join(" ")) if decoration

    @convertWeight: (weight)->
      return Math.round((800*(weight-25)/(87-25)+100)/100)*100

    @applyTextColorInfo: (txtElem, colorData)->
      parts = _.str.words(colorData, ',')
      return unless parts.length == 3 or parts.length == 4
      rgba = parts.join(',')
      txtElem.css('color', "rgba(#{rgba})")
    
  class SystemEventSingleton
    instance = null

    @instance: ->
      instance ?= new SystemEvent
      
    class SystemEvent extends Backbone.Model
      initialize: ->
        @prevWeekDay = (new Date()).getDay()
        setTimeout(@onUpdateTimeEvent, 60*1000)

      onUpdateTimeEvent: =>
        curWeekDay = (new Date()).getDay()
        return if curWeekDay == @preWeekDay

        @preWeekDay = curWeekDay
        @trigger("weekDayChanged")
        setTimeout(@onUpdateTimeEvent, 60*1000)

  class ScriptLoaderSingleton
    instance = null

    @instance: ->
      instance ?= new ScriptLoader

    class ScriptLoader
      load: (scriptUrl, done, failed)->
        console.debug("start to load " + scriptUrl)
        @loadingStatus ?= {}

        stateData = @loadingStatus[scriptUrl]
        if typeof stateData == 'undefined'
          @loadingStatus[scriptUrl] = {state: 'loading', done: [], failed: []}
          dataType = _.str.endsWith(scriptUrl, ".js") and "script" or "text"
          $.ajax(
            url: scriptUrl
            dataType: dataType
            cache: true
            success: (data, status, xhr)->
              # inject returned css data
              return unless _.str.endsWith(scriptUrl, ".css")
              $('<style type="text/css"></style>').html(data).appendTo("head")
          ).done(=>
            @scriptLoaded(scriptUrl)
          ).fail(=>
            @scriptFailed(scriptUrl)
          )
          @loadingStatus[scriptUrl].done.push done if done
          @loadingStatus[scriptUrl].failed.push failed if failed
        else if stateData.state == 'loaded'
          done(scriptUrl) if done
        else if stateData.state == 'failed'
          failed(scriptUrl) if failed
        else if stateData.state == 'loading'
          stateData.done.push done if done
          stateData.failed.push failed if failed
          @loadingStatus[scriptUrl] = stateData

      scriptLoaded: (scriptUrl)=>
        stateData = @loadingStatus[scriptUrl]
        if typeof stateData == 'undefined'
          console.error "can not find loading status data for " + scriptUrl
          return

        stateData.state = 'loaded'
        done(scriptUrl) for done in stateData.done
        stateData.done = []
        stateData.failed = []
        @loadingStatus[scriptUrl] = stateData

      scriptFailed: (scriptUrl)=>
        stateData = @loadingStatus[scriptUrl]
        if typeof stateData == 'undefined'
          console.error "can not find loading status data for " + scriptUrl
          return

        stateData.state = 'failed'
        failed(scriptUrl) for failed in stateData.failed
        stateData.done = []
        stateData.failed = []
        @loadingStatus[scriptUrl] = stateData

  ##########################################################
  # DATA MODELS
  ##########################################################
  class SlotType extends Backbone.Model
    idAttribute: 'name'

    needUI: ->
      @isParameterRequired() or @isConfirmRequired()

    rangeLabels: ->
      _.str.words(@get('rangeLabels'), ',')

    isNumber: ->
      @get("type") in ['byte', 'short', 'int', 'long', 'float', 'double']

    isBoolean: ->
      @get("type") == 'bool'

    isIntegralNumber: ->
      @get("type") in ['byte', 'short', 'int', 'long']

    isFloatNumber: ->
      @get("type") in ['float', 'double']

    isConfirmRequired: ->
      _.str.toBoolean(@get('isConfirmRequired'))

    isParameterRequired: ->
      _.str.toBoolean(@get('isParameterRequired'))
      
    isAllowNull: ->
      _.str.toBoolean(@get('isAllowNull'))


  class SlotTypes extends Backbone.Collection
      model: SlotType

  class CompTypeInfo extends Backbone.Model

    initialize: ->
      @properties = new SlotTypes()
      @actions = new SlotTypes()

    build: (node)->
      @id = node.getAttribute("typeName")
      propertiesNode = node.getElementsByTagName("properties")[0]
      @buildProperties(propertiesNode) if propertiesNode

      actionsNode = node.getElementsByTagName("actions")[0]
      @buildActions(actionsNode) if actionsNode

    buildProperties: (propsNode)->
      for propNode in propsNode.getElementsByTagName("property")
        attrs = attrMapListToMap(propNode.attributes)
        @properties.add(attrs)

    buildActions: (actionsNode)->
      for actionNode in actionsNode.getElementsByTagName("action")
        attrs = attrMapListToMap(actionNode.attributes)
        @actions.add(attrs)

    getAction: (slotName)->
      @actions.get(slotName)

    getProperty: (slotName)->
      @properties.get(slotName)


  class CompTypeInfosSingleton
    instance = null

    @instance: ->
      instance ?= new CompTypeInfos

    class CompTypeInfos extends Backbone.Collection
      model: CompTypeInfo

      build: (node)->
        for infoNode in node.getElementsByTagName("compTypeInfo")
          info = new CompTypeInfo()
          info.build(infoNode)
          @add(info)

  class Slot extends Backbone.Model
    idAttribute: 'slotName'
    url: ->
      @get('slotName')

    getValue: ->
      value = @get('value')
      type = @get('type')
      switch type
        when 'float', 'double'
          return parseFloat(value)
        when 'int', 'long', 'short'
          return parseInt(value)
        else
          return value

  class Slots extends Backbone.Collection
    model: Slot

  class Component extends Backbone.Model
    idAttribute: 'compPath'

    initialize: ->
      @slots = new Slots
      @listenTo(@, 'change:slots', @onSlotsChanged)
      @fullLoad = true

    addSlot: (slotParams)->
      @slots.add(slotParams, {merge: true})
      @slots.get(slotParams.slotName)

    getSlot: (slotName)->
      @slots.get(slotName)

    isApp: ->
      return @id == '/'

    url: (forceFull=false, usedCompSlots=null)->
      # # everytime use the object's URL, so that easyioCpt kit side will not
      # # need to handle the slots part in URL(too many slots(32+) will cause
      # # buffer overflow).
      # return @id

      if @fullLoad
        @fullLoad = false
        return @id
      else if forceFull
        return @id
      else
        # TODO: handle relative data path
        # return ("#{@id}.#{slot.url()}" for slot in @slots.models when @slotTypeFor(slot) == 'property').join('~')
        # compact url for a component object
        slotUrls = []
        for slot in @slots.models
          continue if @slotTypeFor(slot) != 'property'
          slotUrls.push slot.url() if _.isNull(usedCompSlots) or _.contains(usedCompSlots, @id + "." + slot.url())

        # slotUrls = (slot.url() for slot in @slots.models when @slotTypeFor(slot) == 'property')
        return if slotUrls.length == 0 # no slot available
        # return @id if slotUrls.length > 6 # use get full component's content to avoid too many slots to cause too many requests
        return @id + '.' + slotUrls.join('%2B') 

    compTypeInfo: ->
      return null unless @has('typeName')
      compTypeInfo = CompTypeInfosSingleton.instance().get(@get('typeName'))
      unless compTypeInfo
        console.warn("can not find TypeInfo for type: #{@get('typeName')}")
        return null
      compTypeInfo

    slotTypeFor: (slot)->
      typeInfo = @compTypeInfo()
      return 'unknown' unless typeInfo?

      if typeInfo.properties.get(slot.id)
        return 'property'
      else if typeInfo.actions.get(slot.id)
        return 'action'
      else
        # console.warn("unknown slot type for: #{@get('typeName')}.#{slot.id}")
        return 'unknown'

    urlFor: (slot)->
      if (_.isString(slot))
        "#{@id}.#{slot}"
      else
        "#{@id}.#{slot.url()}"

    onSlotsChanged: (model, value, options) =>
      # GOTCHA: Components collection will parse response and add slots as
      # attribute to component object, this is a workaround to move slots
      # from attributes to @slots collection, so that we can use event easier
      @slots.set(@attributes.slots, {merge: true, remove: false})
      delete @attributes.slots

  class ComponentsPool
    instance = null
    proxyUrl = "data_api.php"
    urlBase = "/app/objects"

    @instance: ->
      instance ?= new Components

    class Components extends Backbone.Collection
      model: Component

      contructor: ->

      initialize: ->
        @batchReqUrls = []
        @_maxUrlLength = null
        @usedCompSlots = []

      useCompSlot: (path)->
        @usedCompSlots.push(path) unless _.contains(@usedCompSlots, path)

      maxUrlLength: ->
        if @_maxUrlLength == null
          @_maxUrlLength = parseInt(getUrlParam("maxUrlLength"))
          @_maxUrlLength = 1000 if isNaN(@_maxUrlLength)

        Math.min(@_maxUrlLength, 2000) # should < 2048, the limit of easyioCpt kit

      proxyUrl: ->
        proxyUrl
        
      isCompUrlEmpty: ->
        _.isEmpty(_.compact(comp.url(false, @usedCompSlots) for comp in @models))

      isBatchReqDone: ->
        @batchReqUrls.length == 0

      collectCompUrls: (full=false)->
        result = []

        urls = _.compact(comp.url(full, @usedCompSlots) for comp in @models)
        return result if _.isEmpty(urls)

        for url in urls
          if result.length == 0
            result.push(url)
            continue

          oneBatch = result[result.length-1] 
          if (oneBatch.length + url.length + 1) <= @maxUrlLength()
            oneBatch += '~' + url
            result[result.length-1] = oneBatch
          else
            result.push(url)
        return result

      buildBatchReqUrls: ->
        return if @batchReqUrls.length > 0

        # NOTE: use the url format
        # 'comp1.slot1+slot2+slot3...~comp2.slot1+slot2+slot3...' first, it
        # generates too many batches, switch to another format:
        # 'comp1~comp2...', in order to avoid too many data query requests
        @batchReqUrls = @collectCompUrls()
        if (@batchReqUrls.length > 2)
          @batchReqUrls = @collectCompUrls(true)

        @batchReqUrls

      # GOTCHA: NEVER call this method directly!!!
      # this method has some side effect, the first time call and later
      # calls will return different result(the first time is a full load)
      url: ->
        @buildBatchReqUrls()
        compUrls = @batchReqUrls.shift()
        compUrls = "/#{compUrls}" unless _.str.startsWith(compUrls, '/')
        "#{proxyUrl}?url=#{urlBase}#{compUrls}"

      urlFor: (comp, slot)->
        slotUrl = comp.urlFor(slot)
        slotUrl = "/#{slotUrl}" unless _.str.startsWith(slotUrl, '/')
        "#{urlBase}#{slotUrl}"

      buildUrl: (compPath, slotPath)->
        result = "#{urlBase}#{compPath}"
        result = result + ".#{slotPath}" if slotPath?
        result

      removeCompStartsWith: (pathPrefix)->
        return if _.str.isBlank(pathPrefix)
        @remove(_.filter(@models, (model)->
          model.id == pathPrefix or _.str.startsWith(model.id, "#{pathPrefix}/")
        ))

      handleError: (response)->
        result = []

        if _.str.toNumber(response.resultCode) != -1
          console.error "got an error response:"
          console.error response
          return result

        invalidPaths = []
        for error in response.errors
          unless error.path?
            console.error "can not handle error: #{error}"
            continue
          invalidPaths.push(error.path)

        @removeCompStartsWith(path) for path in _.uniq(invalidPaths)
        return []

      parse: (orig_response)->
        redirect(orig_response.redirect) if orig_response.redirect?

        result = []
        if orig_response.error
          view = new AlertMessageView
          view.render(L("Error"), orig_response.error.text)
          return result

        response = orig_response.response
        if response.resultCode? and response.resultCode != 0
          result = @handleError(response)
        else
          unless _.isArray(response.data)
            console.error "response.data is not a valid array"
            # @trigger("errorResponse", "data format error")
            return result

          result = _.chain(response.data)
            .reject (compData) =>
              !compData.path?
            .map (compData) =>
              compData.compPath = compData.path
              delete compData.path
              _.map compData.slots, (slotData) =>
                slotData.slotName = slotData.name
                delete slotData.name
                slotData
              compData
            .value()
        @trigger("dataload") if @isBatchReqDone()
        return result

      compByCompPath: (compPath)->
        ComponentsPool.instance().get(compPath2compId(compPath))

  # GRAPHICS MODELS
  class GrProperty extends Backbone.Model
    idAttribute: 'name'

    # build: (node)->
    #   name = propertyNode.getAttribute('name')
    #   value = propertyNode.getAttribute('value')
    #   @set(name, value)
    #   if propertyNode.hasAttribute("_rect")
    #     rect = propertyNode.getAttribute("_rect")
    #     @set("_#{name}_rect", rect)

  class GrProperties extends Backbone.Collection
    model: GrProperty

    initialize: ->
      @listenTo(@, 'add', @onPropertyAdded)

    build: (propertiesNode) ->
      for propertyNode in propertiesNode.getElementsByTagName("property")
        attrs = attrMapListToMap(propertyNode.attributes)
        @add(attrs)

    propVal: (name, attr=null)->
      attr = 'value' unless attr
      @get(name)?.get(attr)
      
    userProps: ->
      @.chain().filter((prop)->
        isUserProp(prop.id)
      ).map((prop)->
          prop.id
      ).value()
      
    hasProp: (name)->
      @.some (prop)->
        prop.id == name

    onPropertyAdded: (model, collection, options)=>
      @listenTo(model, 'change', =>
        @trigger('change')
      )

    onValueEventAdded: (model, collection, options)=>
      @listenTo(model, 'propChanged', @onPropChanged)
      @listenTo(model, 'dataOutOfBound', @onDataOutOfBound)

    onPropChanged: (name, value)=>
      console.info "prop #{name} changed to #{value}"
      @get(name)?.set('value', value)

    onDataOutOfBound: (name, value)=>
      console.info "data changed to #{value} but without matched prop #{name} changed"
      @trigger("dataOutOfBound:#{name}", name, value)
      # need to clear prop's value so that next time when its value becomes
      # valid, the prop changed event will be triggered
      @get(name)?.set('value', null)

  class GrRule extends Backbone.Model

    getPropVal: (propName, data)->
      propVal = @get(propName)
      if propName == 'text'
        label = @get('Symbol_label') ? @get('State_label') ? data
        propVal = _.str.sprintf(propVal, label)
      else if propName == 'image'
        propVal = @get("pngAnimation") ? propVal
      propVal

    isMatch: (slot) ->
      return false unless (value = slot.get('value'))?
      dataType = @collection.dataType
      value = slot.get('value')
      switch dataType
        when 'float', 'double'
          return @checkNumberVal(value)
        when 'int', 'long', 'byte'
          if @collection.rangeLabels?
            return @checkEnum(value)
          else
            return @checkNumberVal(value)
        when 'bool'
          return @checkBoolVal(value)
        when 'str'
          return @checkStrVal(value)
        else
          console.error "unknown datatype: #{dataType}"
          return false

    checkBoolVal: (value)->
      boolVal = boolToNumber(value)
      return boolVal == _.str.toNumber(@get('State'))

    checkNumberVal: (value)->
      min = _.str.toNumber(@get("Min"))
      max = _.str.toNumber(@get("Max"))
      if _.str.isBlank(@get('Max')) and _.str.isBlank(@get('Min'))
        return true
      if _.str.isBlank(@get('Max'))
        return value >= min
      else if _.str.isBlank(@get('Min'))
        return value < max
      else
        if max < min
          [max, min] = [min, max]
        if max == min
          return value == max
        else
          return min <= value < max

    checkEnum: (value)->
      if _.isString(value)
        return _.str.toNumber(value) == _.str.toNumber(@get('Symbol'))
      else
        return value == _.str.toNumber(@get('Symbol'))
  
    checkStrVal: (value)->
      func = @get("Function_label")
      expected = @get("Parameter")
      if func == "exactMatch"
        return value == expected
      else if func == "contains"
        return _.str.include(value, expected)
      else if func == "startsWith"
        return _.str.startsWith(value, expected)
      else if func == "endsWith"
        return _.str.endsWith(value, expected)
      else
        console.warn("invalid function: " + func)
        return false

  class GrRules extends Backbone.Collection
    model: GrRule

    initialize: ->
      @dataType = 'float'
      @rangeLabels = null
      @boolLabels = null
      @userDefineFunc = null

    build: (rulesNode) ->
      for ruleNode in rulesNode.getElementsByTagName("rule")
        @add(attrMapListToMap(ruleNode.attributes))

  class GrProcessor extends Backbone.Model

    formatValue: (data)->
      try
        @doFormatValue(data)
      catch err
        console.warn "get some error: #{err.message}"
        formatString = @get('formatString')
        console.debug "format string: #{formatString}"
        ""

    doFormatValue: (data)->
      formatString = @get('formatString')
      dataType = @get('dataType')
      if dataType == 'bool'
        labels = @get('boolLabels')?.split(',')
        if labels?
          boolVal = boolToNumber(data)
          _.str.sprintf(formatString, labels[boolVal])
        else
          _.str.sprintf(formatString, data)
      else if dataType == 'sys::Buf' or dataType == 'str'
        _.str.sprintf(formatString, data)
      else
        labels = @get('rangeLabels')?.split(',')
        offset = _.str.toNumber(@get("rangeOffset")) ? 0
        data -= offset
        if labels?
          _.str.sprintf(formatString, labels[data])
        else
          _.str.sprintf(formatString, data)

    process: (data)->
      # NOTE: this method is buggy!
      switch @get('value')
        when 'ConvertToString'
          @formatValue(data)
        when 'UserDefine'
          jsCode = @get('userDefineCodes')
          return data if _.str.isBlank(jsCode)
          proxyObject = {data: data}
          try
            unless @userDefineFunc
              @userDefineFunc = new Function(jsCode + "; return process(this.data);")
            return @userDefineFunc.call(proxyObject)
          catch error
            console.warn("failed to execute user define processor")
            return data
        else
          data

    isProcessorEnabled: ->
      @get('value') == 'PassThrough'

  class GrDataSource extends Backbone.Model

    isEmpty: ->
      not (@get('compPath')? and @get('slotName')?)

    component: ->
      ComponentsPool.instance().compByCompPath(@get('compPath'))

    compPath: ->
      @get('compPath')

    slotName: ->
      @get('slotName')

    fullPath: ->
      @get('compPath') + "." + @get('slotName')

    isDataBound: ->
      @component()?

    compTypeInfo: ->
      CompTypeInfosSingleton.instance().get(@get('typeName'))

  class GrValueEvent extends Backbone.Model
    idAttribute: 'propName'

    initialize: ->
      @grDataSource = new GrDataSource
      @grProcessor = new GrProcessor
      @grRules = new GrRules

    build: (valueEventNode) ->
      @set(attrMapListToMap(valueEventNode.attributes))

      dataSourceNode = valueEventNode.getElementsByTagName("dataSource")[0]
      @grDataSource.set(attrMapListToMap(dataSourceNode.attributes))
      return if @grDataSource.isEmpty()

      comp = @getComp()
      slot = comp.addSlot(_.pick(@grDataSource.attributes, 'slotName'))
      @listenTo(slot, 'change', @applyValueEvent)

      processorNode = valueEventNode.getElementsByTagName("processor")[0]
      @grProcessor.set(attrMapListToMap(processorNode.attributes))

      rulesNode = valueEventNode.getElementsByTagName("rules")[0]
      @grRules.dataType = @grProcessor.get('dataType')
      @grRules.rangeLabels = @grProcessor.get('rangeLabels')
      @grRules.boolLabels = @grProcessor.get('boolLabels')
      @grRules.build(rulesNode) if rulesNode?

      compPool = ComponentsPool.instance()
      compPool.useCompSlot(@grDataSource.fullPath())

      @

    getComp: () ->
      compPool = ComponentsPool.instance()

      # GOTCHA: remove app object's name
      compPath = compPath2compId(@grDataSource.get('compPath'))
      compPool.add({compPath: compPath, typeName: @grDataSource.get('typeName')}, {merge: true})
      compPool.get(compPath)

    applyValueEvent: (slot, options) =>
      propName = @get('propName')
      data = slot.getValue()

      data = @grProcessor.process(data)

      eventTriggered = false
      for rule in @grRules.models
        continue unless rule.isMatch(slot)
        @trigger('propChanged', propName, rule.getPropVal(propName, data))
        eventTriggered = true
        break

      if not eventTriggered
        if @grProcessor.isProcessorEnabled() and @grRules.models.length > 0
          @trigger('dataOutOfBound', propName, data)
        else
          @trigger('propChanged', propName, data)

    dataSource: ->
      propName = @get('propName')
      return null unless (propName == 'image' or propName == 'text' or propName == 'inputData' or propName == 'visible' )
      if @grDataSource.isEmpty() then null else @grDataSource

  class GrValueEvents extends Backbone.Collection
    model: GrValueEvent

    build: (valueEventsNode) ->
      for valueEventNode in valueEventsNode.getElementsByTagName("valueEvent")
        grValueEvent = new GrValueEvent
        @add(grValueEvent) if grValueEvent.build(valueEventNode)?

  class GrMenuAction extends Backbone.Model

    build: (menuActionNode) ->
      @set(attrMapListToMap(menuActionNode.attributes))

  class GrMenuActions extends Backbone.Collection
    model: GrMenuAction

    build: (menuActionsNode) ->
      for menuActionNode in menuActionsNode.getElementsByTagName("menuAction")
        grMenuAction = new GrMenuAction
        @add(grMenuAction) if grMenuAction.build(menuActionNode)?

    hasVisibleAction: ->
      return false if @length <= 0
      for action in @models
        if action.get('show') == '1'
          return true
      false

  class GrObject extends Backbone.Model
    initialize: ->
      @grProperties = new GrProperties
      @grValueEvents = new GrValueEvents
      @grMenuActions = new GrMenuActions
      @grProperties.listenTo(@grValueEvents, 'add', @grProperties.onValueEventAdded)

    build: (grObjectNode) ->
      @set(attrMapListToMap(grObjectNode.attributes))

      propertiesNode = grObjectNode.getElementsByTagName("properties")[0]
      @grProperties.build(propertiesNode) if propertiesNode?

      valueEventsNode = grObjectNode.getElementsByTagName("valueEvents")[0]
      @grValueEvents.build(valueEventsNode) if valueEventsNode?

      menuActionsNode = grObjectNode.getElementsByTagName("menuActions")[0]
      @grMenuActions.build(menuActionsNode) if menuActionsNode?

    isPropertyBound: (propName)->
      return false unless @grValueEvents.get(propName)?
      return not @grValueEvents.get(propName).grDataSource.isEmpty()

    dataSource: ->
      for index in [0...@grValueEvents.length]
        dataSource = @grValueEvents.at(index).dataSource()
        return dataSource if dataSource?
      null

    isVisible: ->
      _.str.toBoolean(@grProperties.propVal('visible'))

    isDataValid: ->
      return true unless @dataSource()?
      @dataSource().isDataBound()

  class GrObjects extends Backbone.Collection
    model: GrObject

  class GrLayer extends Backbone.Model
    initialize: ->
      @grObjects = new GrObjects

    build: (layerNode) ->
      for grObjectNode in layerNode.getElementsByTagName("grObject")
        grObject = new GrObject
        grObject.build(grObjectNode)
        @grObjects.add(grObject, {validate:false})

  class GrLayers extends Backbone.Collection
    model: GrLayer

    build: (layersNode) ->
      for layerNode in layersNode.getElementsByTagName("layer")
        layer = new GrLayer
        layer.build(layerNode)
        @add(layer)

  class GrCanvas extends Backbone.Model
    initialize: ->
      @grLayers = new GrLayers

    build: (canvasNode) ->
      @set(attrMapListToMap(canvasNode.attributes))

      layersNode = canvasNode.getElementsByTagName("layers")[0]
      return unless layersNode?

      @grLayers.build(layersNode)

  class GrDoc extends Backbone.Model

    initialize: ->
      @actionPermitted = true
      @grVersion = 0.0

    url: ->
      "grdata.php?grName=#{encodeURIComponent(GrNavInstance.instance().grName())}"

    fetchDocInfo: =>
      $.ajax
        type: "GET"
        url: 'grdata.php'
        data:
          grName: encodeURIComponent(GrNavInstance.instance().grName())
          infoOnly: true
        dataType: 'json'
        success: (data)=>
          if @parseDocInfo(data)
            @trigger("docInfoLoaded")
          else
            view = new AlertMessageView
            view.render(L("Error"), L("can not parse graphic document info data"))
        error: (data, status)=>
          console.log "error status: " + status
          view = new AlertMessageView
          view.render(L("Error"), L("can not fetch graphic document info: ") + status)
        complete: (data, status)->
          console.log "[GrDoc] request completed, data: #{data}, status: #{status}"
        
    fetch: (options) ->
      options ?= {}
      options.dataType = 'xml'
      super options

    parseDocInfo: (response) ->
      return false unless response?

      redirect(response.redirect) if response.redirect?

      data = null
      if response.error
        view = new AlertMessageView
        view.render(L("Error"), response.error.text)
        return false
      else
        @actionPermitted = _.str.toBoolean(response.actionPermitted)
        return true

    parse: (response, options) ->
      result = {}
      return result unless response?

      # data = $.parseXML(response)
      data = response
      if not data
        view = new AlertMessageView
        view.render(L("Error"), L("can not parse gr file {grName}").supplant(grName: GrNavInstance.instance().grName()))
        return result

      canvasNode = data.getElementsByTagName("canvas")[0]
      return result unless canvasNode?

      for attr in canvasNode.attributes
        continue unless attr.name == 'version'
        @grVersion = _.str.toNumber(attr.value)
        break

      result.grCanvas = new GrCanvas()
      result.grCanvas.build(canvasNode)

      compTypeInfosNode = data.getElementsByTagName("compTypeInfos")[0]
      return result unless compTypeInfosNode?
      result.compTypeInfos = CompTypeInfosSingleton.instance()
      result.compTypeInfos.build(compTypeInfosNode)

      result

    version: ->
      @grVersion

  class GrItem extends Backbone.Model

    isGroup: ->
      false

    path: ->
      @get('path')

    isHome: ->
      not _.str.isBlank(@get('home'))

    icon: ->
      if @isGroup()
        'icon-book'
      else
        GrNavInstance.instance().homePage() == @path() and 'icon-home' or 'icon-picture'

  class GrGroup extends GrItem

    initialize: ->
      @items = []

    isGroup: ->
      true

    addItem: (item)->
      @items.push item

    contains: (path)->
      for item in @items
        if item.isGroup()
          return true if item.contains(path)
        else
          return true if item.path() == path
      false

    defaultHomePage: ->
      homePage = null
      for item in @items
        if item.isGroup()
          homePage = item.defaultHomePage()
        else
          homePage = item.path() if item.isHome()
        break unless _.str.isBlank(homePage)
      homePage

    firstAccessablePage: (blackList)->
      page = null
      for item in @items
        if item.isGroup()
          page = item.firstAccessablePage(blackList)
        else
          page = item.path()
        break unless _.contains(blackList, page)
      page

  class GrNavInstance
    instance = null

    class GrNavModel extends Backbone.Model
      url: "grdata.php?grName=grNav.xml"

      initialize: ->
        @topGroup = null
        @personalHomePage = null
        @grBlackList = []
        @firstGrPath = null
        @dateTimeObjPath = null

      homePage: ->
        # return personal home page if it's valid
        return @personalHomePage if (not _.str.isBlank(@personalHomePage)) and @contains(@personalHomePage)

        # otherwise, return the default global home page if it's valid
        defaultHomePage = @topGroup.defaultHomePage()
        if not _.str.isBlank(defaultHomePage) and @contains(defaultHomePage)
          return defaultHomePage

        return null
    
      showAtStart: ->
        return true unless @topGroup.has('visibilityAtStart')
        return @topGroup.get('visibilityAtStart') == "show"

      grName: ->
        grname = getUrlParam("grname")
        return grname if grname?

        grname = @homePage()
        return grname if grname? and (not _.contains(@grBlackList, grname))

        @topGroup.firstAccessablePage(@grBlackList)

      parse: (response, options) ->
        result = {}
        return result unless response?

        redirect(response.redirect) if response.redirect?

        if response.error
          view = new AlertMessageView
          view.render(L("Error"), response.error.text)
          return result
        else
          @personalHomePage = response.home_page
          if response.grBlackList?
            @grBlackList = _.str.words(response.grBlackList, ',')
          data = $.parseXML(response.data)

        rootNode = data.getElementsByTagName("grNav")[0]
        return result unless rootNode?

        groupNode = childElementsByTagName(rootNode, "grGroup")[0]
        return result unless groupNode?

        @topGroup = @parseGroup(groupNode)
        firstItem = data.getElementsByTagName("grItem")[0]
        @firstGrPath = firstItem.getAttribute("path")

        dateTimeNode = data.getElementsByTagName("dateTime").item(0)
        @dateTimeObjPath = dateTimeNode?.getAttribute("compPath")

        {}

      parseGroup: (groupNode)->
        group = new GrGroup(domAttrsToMap(groupNode))
        for itemNode in childElementsByTagName(groupNode, "grItem", "grGroup")
          if itemNode.tagName == "grItem"
            group.addItem(new GrItem(domAttrsToMap(itemNode)))
          else
            group.addItem(@parseGroup(itemNode))
        return group

      contains: (path)->
        @topGroup?.contains(path)

    @instance: ->
      instance ?= new GrNavModel

  # PERMISSION DATA MODELS
  class UserPermission extends Backbone.Model
    idAttribute: 'name'

    url: ->
      "permission.php?user_id=#{@get('user_id')}"

    initialize: ->
      @topGroup = grNavInstance.topGroup
      @perms = null

    home_page: ->
      personalHomePage = @get('home_page')
      if grNavInstance.contains(personalHomePage)
        return personalHomePage
      else
        return grNavInstance.homePage() or grNavInstance.firstGrPath

    permissions: ->
      if _.isNull(@perms)
        @perms = []
        @buildPermissions(@topGroup, 1)

      @perms

    isDevReadable: ->
      @get('dev_readable') == 't'
      
    isDevWritable: ->
      @get('dev_writable') == 't'

    iconFor: (perm)->
      isHome = perm.path == @home_page()
      isHome and 'icon-home' or 'icon-picture'

    isNoAccess: (perm)->
      not (perm.readable and perm.writable)

    isReadOnly: (perm)->
      perm.readable and (not perm.writable)

    isReadWrite: (perm)->
      perm.writable

    buildPermissions: (groupNode, level)->
      for item in groupNode.items
        orig_perms = @get("permissions") ? []
        blackPerm = _.find(orig_perms, (d)=>
          d.path == item.get('path')
        )
        readable = not (blackPerm?.readable? and blackPerm.readable in ['f', 'false'])
        writable = not (blackPerm?.writable? and blackPerm.writable in ['f', 'false'])
        @perms.push({
          name: item.get('name'),
          path: item.get('path'),
          readable: readable,
          writable: writable,
          isGroup: item.isGroup(),
          indent: 8*level})
        @buildPermissions(item, level+1) if item.isGroup()

    updateData: (path, permStr)->
      for perm, index in @perms
        continue unless perm.path == path

        if permStr == 'noaccess'
          @perms[index].readable = false
          @perms[index].writable = false
        else if permStr == 'readonly'
          @perms[index].readable = true
          @perms[index].writable = false
        else if permStr == 'readwrite'
          @perms[index].readable = true
          @perms[index].writable = true
        else
          console.warn("invalid perm string: " + permStr)
        break

    updateDevPerm: (dev_readable, dev_writable)->
      @set("dev_readable", if dev_readable then 't' else 'f')
      @set("dev_writable", if dev_writable then 't' else 'f')

    postData: ->
      perm_data = []
      for perm in @perms
        continue if perm.readable and perm.writable
        perm_data.push(_.pick(perm, 'path',
                               'readable', 'writable'))
      {
        user_id: @get('user_id'), home_page: @get('home_page'),
        dev_readable: @get('dev_readable'), dev_writable: @get('dev_writable'),
        perms: perm_data
      }

    # can: (action, path)->
    #   perms = @permissions()
    #   for perm in perms
    #     continue if perm.path != path

    #     readable = perm.readable not in ['f', 'false']
    #     writable = perm.writable not in ['f', 'false']
    #     if action == 'read'
    #       return readable
    #     else
    #       return readable and writable
    #   return true

  class UserPermissionsSingleton
    instance = null

    @instance: ->
      instance ?= new UserPermissions

    class UserPermissions extends Backbone.Collection
      model: UserPermission

      url: (user_id=null)->
        if user_id
          "permission.php?user_id=#{user_id}"
        else
          'permission.php'

      parse: (response)->
        redirect(response.redirect) if response.redirect?

        result = _.chain(response.data)
          .reject (r)->
            _.str.isBlank(r.name)
          .groupBy (r)->
            r.name
          .value()

        result = _.map _.pairs(result), (r)->
          user_id = r[1][0].user_id
          home_page = r[1][0].home_page
          dev_readable = r[1][0].dev_readable
          dev_writable = r[1][0].dev_writable
          perms = _.reject r[1], (item)->
            _.str.isBlank(item.path)
          {'name': r[0], 'user_id': user_id, 'home_page': home_page,
          'dev_readable': dev_readable, 'dev_writable': dev_writable,
          'permissions': perms}

        # result = _.sortBy(result, 'user_id')
        result

  # ACCOUNT DATA MODELS
  class Account extends Backbone.Model
    idAttribute: 'name'

    url: ->
      "account_management.php?user_id=#{@get('user_id')}"

    enableUtility: (enabled)->
      if enabled
        @set('utility_enabled', 't')
      else
        @set('utility_enabled', 'f')

    enableSystem: (enabled)->
      if enabled
        @set('system_enabled', 't')
      else
        @set('system_enabled', 'f')
        
    enableAccountManagement: (enabled)->
      if enabled
        @set('account_management_enabled', 't')
      else
        @set('account_management_enabled', 'f')
        
    enablePasswordChange: (enabled)->
      if enabled
        @set('password_change_enabled', 't')
      else
        @set('password_change_enabled', 'f')
        
    enableDashboard: (enabled)->
      if enabled
        @set('dashboard_enabled', 't')
      else
        @set('dashboard_enabled', 'f')
        
    enableDashboardLanding: (enable)->
      if enable
        @set('dashboard_as_landing_page', 't')
      else
        @set('dashboard_as_landing_page', 'f')

    postData: ->
      system_enabled: @get('system_enabled')
      utility_enabled: @get('utility_enabled')
      account_management_enabled: @get('account_management_enabled')
      password_change_enabled: @get('password_change_enabled')
      dashboard_enabled: @get('dashboard_enabled')
      dashboard_as_landing_page: @get('dashboard_as_landing_page')

  class AccountsSingleton
    instance = null

    @instance: ->
      instance ?= new Accounts

    class Accounts extends Backbone.Collection
      model: Account

      url: (user_id=null)->
        if user_id
          "account_management.php?user_id=#{user_id}"
        else
          'account_management.php'

      parse: (response)->
        redirect(response.redirect) if response.redirect?
        response.data

  # Chart & Table data models
  class LineData extends Backbone.Model
    idAttribute: 'name'

    updateValue: (val, keepDataOfLastNMin)->
      now = (new Date()).getTime()
      lastUpdateTimeMs = @get('lastUpdateTimeMs') ? now
      elapsedMs = now - lastUpdateTimeMs

      data = @get('data')
      data = _.reject(data, (d)=>
        d[0] -= elapsedMs
        d[0] < -1*(keepDataOfLastNMin*60 + 10)*1000
      )

      precision = 0
      dotPos = val.indexOf('.')
      if dotPos != -1
        precision = Math.max(precision, val.length - dotPos - 1)
      data.push([0, _.str.toNumber(val, precision)])
      @set({'data': data, 'lastUpdateTimeMs': now})

    latestVal: ->
      _.last(@get('data'))[1]

  class LineDatas extends Backbone.Collection
    model: LineData

    initialize: ->
      @propertySet = null
      @keepDataOfLastNMin = 20

    datas: ->
      _.map(@models, (m)->
        m.attributes
      )

    updateDatas: ->
      _.each(@models, (m)=>
        val = @propertySet.propVal(m.id)
        m.updateValue(val, @keepDataOfLastNMin)
      )

  class HistoryLineData extends Backbone.Model
    idAttribute: 'name'

  class HistoryLineDatas extends Backbone.Collection
    model: HistoryLineData

    initialize: ->
      @propertySet = null

      @timeRange =
        min: null
        max: null

      @valRange =
        min: null
        max: null

      @dt_after = null
      @dt_before = null

      @queryParams =
        table: null
        columns: null
        dt_after: null
        dt_before: null
        dt_tz: jstz.determine().name()
        format: 'json'
        file: 0

    prepareParams: ->
      return false if _.isNull(@dt_after) or _.isNull(@dt_before)
      table = @propertySet.propVal('tableName')
      return false if _.str.isBlank(table)

      @queryParams.table = table
      columns = [m.id for m in @models]
      columns.unshift('dt')
      @queryParams.columns = _.str.join(',', columns)
      @queryParams.dt_after = @dt_after.format('YYYYMMDDHHmmss')
      @queryParams.dt_before = @dt_before.format('YYYYMMDDHHmmss')
      return true

    url: ->
      params = _.chain(@queryParams).pairs().map((v)->
        "#{v[0]}=#{encodeURIComponent(v[1])}"
      ).reduce((memo, param)->
        "#{memo}&#{param}"
      "").value()

      "data_exporter.php?#{params}"

    parse: (response, options)->
      redirect(response.redirect) if response.redirect?

      result = []
      if response.error?
        view = new AlertMessageView
        view.render(L("Error"), response.error.text)
        return result

      @resetRanges()

      unless response.data.rows? and response.data.columns?
        return _.map(@datas(), (d)->
          d.data.length = 0
          d
        )

      rows = response.data.rows
      cols = response.data.columns

      buffers = {}
      for row in rows
        rowObj = _.object(cols, row)
        dt = moment(rowObj.dt, 'YYYY-MM-DD HH:mm:ss').unix()*1000

        # @updateTimeRange(dt)

        for col in cols when col isnt 'dt'
          unless _.has(buffers, col)
            buffers[col] = @get(col).attributes
            buffers[col].data.length = 0

          # keep 10 digits decimals in original data
          val = _.str.toNumber(rowObj[col], 10)
          # @updateValRange(val)
          buffers[col].data.push([dt, val])

      _.values(buffers)

    resetRanges: ->
      @timeRange.min = null
      @timeRange.max = null
      @valRange.min = null
      @valRange.max = null

    updateTimeRange: (dt)->
      unless @timeRange.min
        @timeRange.min = dt
      else
        @timeRange.min = Math.min(dt, @timeRange.min)

      unless @timeRange.max
        @timeRange.max = dt
      else
        @timeRange.max = Math.max(dt, @timeRange.max)

    updateValRange: (val)->
      unless @valRange.min
        @valRange.min = val
      else
        @valRange.min = Math.min(val, @valRange.min)

      unless @valRange.max
        @valRange.max = val
      else
        @valRange.max = Math.max(val, @valRange.max)

    datas: ->
      _.pluck(@models, 'attributes')

    updateDatas: (start, end)->
      return if _.isUndefined(start) or _.isUndefined(end)

      @dt_after = start
      @dt_before = end
      unless @prepareParams()
        console.warn("building query params failed")
        return

      @fetch({remove: false})
      
  class ServerStateCheckerSingleton
    instance = null

    @instance: ->
      instance ?= new ServerStateChecker
      
    class ServerStateChecker
      
      check: (timeout=3000, delay=45000)->
        @retryTimes = 100
        @timeout = timeout
        console.debug("wait for " + delay/1000 + " seconds before checking ...")
        _.delay(@doCheck, delay)
            
      doCheck: =>
        console.info("do server check #{ @retryTimes } ...")
        --@retryTimes
        $.ajax(
          url: "./ping.php"
          timeout: @timeout
          beforeSend: => @queryTime = Date.now()
          error: (jq, status)=>
            console.debug("request error: #{status}")
            # for old version, ping.php not exist, will return 404, that means
            # server is ready already
            if jq.status == 404
              $(window).trigger("serverReady")
            else
              if @retryTimes > 0
                delay = @timeout - (Date.now() - @queryTime)
                if delay > 500
                  _.delay(@doCheck, delay)
                else
                  @doCheck()
              else
                $(window).trigger("serverTimeout")
          success: ->
            $(window).trigger("serverReady")
        )
      
  class AppUtiltiySingleton
    instance = null
    @instance: ->
      instance ?= new AppUtility

    class AppUtility
      save: (success, failure)->
        compPool = ComponentsPool.instance()
        path = compPool.buildUrl('/', 'save')
        @sendRequest({path: path}, success, failure)
        
      reboot: (success, failure)->
        compPool = ComponentsPool.instance()
        path = compPool.buildUrl('/', 'reboot')
        @sendRequest({path: path}, success, failure)

      sendRequest: (data, success, failure)->
        compPool = ComponentsPool.instance()
        data = _.defaults(data, {slotType: 'action', type: 'void', value: ''})

        $.ajax
          type: "POST"
          url: compPool.proxyUrl()
          data: data
          dataType: 'json'
          success: =>
            success and success()
          error: (data, status)=>
            console.log "error status: " + status
            faiurle and failure()
          complete: (data, status)->
            console.log "[AppUtility] request completed, data: #{data}, status: #{status}"

  class DateTimeUtilitySingleton
    instance = null
    @instance: ->
      instance ?= new DateTimeUtility

    class DateTimeUtility extends Backbone.Model
      initialize: ->
        @compPath = GrNavInstance.instance().dateTimeObjPath or '/service/time'
        @data = null
        @refreshStartTime = Date.now()
        @networkDelay = 1000 # default network delay is 1 second
        @startPolling = _.once(@_startPolling)

      refresh: (opts)=>
        compPool = ComponentsPool.instance()
        compUrl = compPool.buildUrl(@compPath)

        @refreshStartTime = Date.now()
        $.ajax
          type: 'GET'
          url: "#{compPool.proxyUrl()}?url=#{compUrl}"
          dataType: 'json'
          success: (data)=>
            result = @onDataLoaded(data)
            if result == false
              opts.error() if opts? and _.isFunction(opts.error)
            else
              opts.success() if opts? and _.isFunction(opts.success)
              @trigger("dateTimeUpdated")
          error: (jq, status, err)=>
            console.warn("failed to get datetime, status: " + status + " error: " + err)
            opts.error() if opts? and _.isFunction(opts.error)
          complete: =>
            opts.complete() if opts? and _.isFunction(opts.complete)
            @networkDelay = (Date.now() - @refreshStartTime)/2

      onDataLoaded: (data)=>
        if data.redirect?
          redirect(data.redirect)
        else
          if data.error
            view = new AlertMessageView
            view.render(L("Error"), data.error.text)
          else
            @handleResponse(data.response)

      handleResponse: (response)->
        if response.resultCode? and response.resultCode != 0
            view = new AlertMessageView
            view.render(L("Error"), L("Can not access data time object, please make sure the datetime object's path({path}) is configured correctly").supplant(path: @compPath))
            return false

        if not _.isArray(response.data) or response.data.length < 1
          console.error "response.data is not a valid array"
          console.debug response.data
          return false

        @data = response.data[0]
        @set(slot.name, slot) for slot in @data.slots

      save: (isOSUTCOffsetMode, tz, utcOffset, sysclockInNs, opts)->
        compPool = ComponentsPool.instance()

        isOSUtcOffsetPath = compPool.buildUrl(@compPath, 'osUtcOffset')
        tzPath = compPool.buildUrl(@compPath, 'tz')
        utcOffsetPath = compPool.buildUrl(@compPath, 'utcOffset')
        setSysClockPath = compPool.buildUrl(@compPath, 'setSysClock')

        @sendRequest("#{isOSUtcOffsetPath}", isOSUTCOffsetMode, 'property', 'bool',
          success: =>
            @sendRequest("#{tzPath}", tz, "property", 'Buf',
              success: =>
                if isOSUTCOffsetMode
                  @sendRequest("#{setSysClockPath}", sysclockInNs, 'action', 'long', opts)
                else
                  @sendRequest("#{utcOffsetPath}", utcOffset, 'property', 'int',
                    success: =>
                      @sendRequest("#{setSysClockPath}", sysclockInNs, 'action', 'long', opts)
                    error: =>
                      opts.error() if opts? and _.isFunction(opts.error)
                  )
              error: =>
                opts.error() if opts? and _.isFunction(opts.error)
            )
          error: =>
            opts.error() if opts? and _.isFunction(opts.error)
        )

      sendRequest: (path, value, slotType, dataType, opts)->
        compPool = ComponentsPool.instance()
        $.ajax
          type: "POST"
          url: compPool.proxyUrl()
          timeout: 30000
          data:
            path: path
            type: dataType
            value: value
            slotType: slotType
          dataType: 'json'
          success: (data)=>
            if data.error
              view = new AlertMessageView
              view.render(L("Error"), data.error.text)
            else if data.redirect?
              redirect(data.redirect)
            else
              opts.success() if opts and _.isFunction(opts.success)
          error: (data, status)=>
            view = new AlertMessageView
            view.render(L("Error"), status)
            opts.error() if opts and _.isFunction(opts.error)
          complete: =>
            opts.complete() if opts and _.isFunction(opts.complete)
        
      _startPolling: =>
        @listenTo(@, "dateTimeUpdated", =>
          _.delay(@refresh, 50*1000)
        )
        @refresh()

  class SettingsSingleton
    instance = null

    @instance: ->
      instance ?= new Settings

    class Settings extends Backbone.Model

      url: ()->
        'settings_controller.php'
        
      ttl_enabled: ->
        @get("ttl_enabled") == 't'
        
      ttl_value: ->
        _.str.toNumber(@get("ttl_value"))
        
      parse: (response)->
        redirect(response.redirect) if response.redirect?

        response.data

  class AuthKeysSingleton
    instance = null
    
    @instance: ->
      instance ?= new AuthKeys

    class AuthKeys extends Backbone.Model

      url: ->
        'auth_key_controller.php'
        
      auth_keys: ->
        @get('data')
        
      addKey: (keyData)->
        keys = @get('data')
        keys.unshift(keyData)
        @set('data', keys)
        
      updateKey: (keyData)->
        keys = @get('data')
        @set('data', _.map(keys, (k)->
          if k.key == keyData.key then keyData else k
        ))
        
      deleteKey: (key)->
        keys = @get('data')
        keys = _.reject(keys, (k)->
          k.key == key
        )
        @set('data', keys)

  ##########################################################
  # GRAPHICS VIEWS
  ##########################################################
  class GrAskParameterDialog extends Backbone.View
    el: "#askParamModal"
    events:
      'click .btn-primary': 'apply'

    initialize: ->
      @menuItemData = {}
      @errors = []
      @title = ''
      @parameterLabel = ''

    render: ->
      if @menuItemData.type == 'action'
        slotType = @menuItemData.slotType
        if !slotType.needUI()
          @sendRequest()
          return true

      @title = @menuItemData.label
      @parameterLabel = L('New Value:')

      @$el.empty()
      @renderHeader()
      @renderContent()
      @renderFooter()
      @$el.find('form').submit(@submitHandler)
      @$el.modal()

    renderHeader: ()->
      $("""
        <div class="alert hide"> </div>
        <div class="modal-header">
          <button type="button" class="close" data-dismiss="modal" aria-hidden="true">&times;</button>
          <h3>#{@title}</h3>
        </div>
        """).appendTo(@$el)

    renderContent: ()->
      slotType = @menuItemData.slotType
      unitLabel = slotType?.get('unitLabel')
      rangeLabels = slotType?.rangeLabels()

      controlHtml = if not _.str.isBlank(unitLabel)
        """
            <label class="control-label" for="slotValue">#{@parameterLabel}</label>
            <div class="controls">
                <div class="input-append">
                  <input class="span2" id="slotValue" type="text" autofocus>
                  <span class="add-on">#{unitLabel}</span>
                </div>
            </div>
        """
      else if not _.isEmpty(rangeLabels)
        options = _.str.toSentence("<option value='#{index}'>#{label}</option>" for label, index in rangeLabels)
        """
            <label class="control-label" for="slotValue">#{@parameterLabel}</label>
            <div class="controls">
              <select id='slotValue'>
                #{options}
              </select>
            </div>
        """
      else if slotType.isBoolean()
        """
            <label class="control-label" for="slotValue">#{@parameterLabel}</label>
            <div class="controls">
              <select id='slotValue'>
                <option value='false'>#{L('false')}</option>
                <option value='true'>#{L('true')}</option>
                <option value='nil'>#{L('null')}</option>
              </select>
            </div>
        """
      else if slotType.isConfirmRequired()
        confirmMsg = L("Are you sure to do '{name}'?").supplant(name: slotType.get('name'))
        """
            <center><h4>#{confirmMsg}</h4></center>
        """
      else
        """
            <label class="control-label" for="slotValue">#{@parameterLabel}</label>
            <div class="controls">
              <input type="text" id="slotValue" autofocus>
            </div>
        """

      $("""
        <div class="modal-body">
          <form class="form-horizontal" >
            <div class="control-group">
              #{controlHtml}
            </div>
          </form>
        </div>
        """).appendTo(@$el)

    renderFooter: ()->
      $("""
        <div class="modal-footer">
          <a href="#" class="btn btn-primary">OK</a>
          <a href="#" class="btn" data-dismiss="modal">Close</a>
        </div>
        """).appendTo(@$el)

    value: ->
      @$el.find("#slotValue").val()

    validateNotEmpty: ->
      if _.str.isBlank(@value())
        @errors.push(L("input can not be empty"))
        false
      else
        true

    validateMaxMinRange: ->
      slotType = @menuItemData.slotType
      return true unless slotType.isNumber()
      return true if not slotType.has('max') and not slotType.has('min')

      max = _.str.toNumber(slotType.get('max'))
      min = _.str.toNumber(slotType.get('min'))
      value = _.str.toNumber(@value())

      return false if not _.isNaN(max) and value > max
      return false if not _.isNaN(min) and value < min
      true

    validateNumber: ->
      return true unless @menuItemData.slotType.isNumber()

      if _.isNaN(_.str.toNumber(@value()))
        if @menuItemData.slotType.isFloatNumber() and (@value() == 'null' or @value() == 'nil')
          true
        else
          @errors.push(L("input is not a valid number"))
          false
      else
        true

    validateIntegralNumber: ->
      return true unless @menuItemData.slotType.isIntegralNumber()

      if /^-?[0-9]+$/.test(@value())
        true
      else
        @errors.push(L("input is not a valid integral number"))
        false

    validate: ->
      slotType = @menuItemData.slotType
      return true unless slotType.isParameterRequired()

      @errors = []

      @validateNotEmpty() and @validateNumber() and @validateMaxMinRange() and @validateIntegralNumber()

    renderError: ->
      @$el.find(".control-group").addClass("error")
      @$el.find(".controls").append("<span class='help-inline'>#{_.str.toSentence(@errors)}</span>")

    submitHandler: (event)=>
      event.preventDefault()
      @apply()

    sendRequest: ->
      compPool = ComponentsPool.instance()
      slotType = @menuItemData.slotType
      comp = @menuItemData.comp
      url = compPool.urlFor(comp, slotType.get('name'))

      value = @value()
      value = "nil" if slotType.isFloatNumber() and value == "null" or value == "nil"

      $.ajax
        type: "POST"
        url: compPool.proxyUrl()
        timeout: 30000
        data:
          path: url
          type: slotType.get('type')
          value: value
          slotType: @menuItemData.type
        dataType: 'json'
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled")
          @$el.find('.btn-primary').attr("disabled", "disabled")
        success: (data)=>
          if data.error
            view = new AlertMessageView
            view.render(L("Error"), data.error.text)
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.modal('hide')
        error: (data, status)=>
          console.log "error status: " + status
          @$el.modal('hide')

          view = new AlertMessageView
          view.render(L("Error"), status)

        complete: (data, status)=>
          @$el.find('.btn-primary').removeClass("disabled")
          @$el.find('.btn-primary').removeAttr("disabled")

    apply: ->
      # clean up possible previous error hints
      @$el.find(".control-group").attr("class", "control-group")
      @$el.find(".controls .help-inline").remove()

      if not @validate()
        @renderError()
        return

      @sendRequest()

  class GrMenuView extends Backbone.View
    el: '#grObjectMenu'

    renderCompMenuActions: (menuAction, menuElem)->
      typeInfos = CompTypeInfosSingleton.instance()
      typeInfo = typeInfos?.get(menuAction.get("typeName"))
      return false unless typeInfo?

      compPool = ComponentsPool.instance()
      comp = compPool.compByCompPath(menuAction.get("compPath"))
      return false unless comp?

      slot = typeInfo.getAction(menuAction.get('slotName'))
      label = menuAction.get('label')
      label = label + '...' if _.str.toBoolean(slot.get('isConfirmRequired')) or _.str.toBoolean(slot.get('isParameterRequired'))
      alink = $("<a tabindex='-1' href='#'>#{ label }</a>")
      alink.data('menuItemData', {type: 'action', slotType: slot, comp: comp, label: label})
      li = $("<li></li>").appendTo(menuElem)
      alink.appendTo(li)
      return true

    renderLinkMenu: (menuElem)->
      link = @model.grProperties.propVal('link')
      return if _.str.isBlank(link)
      
      label = _.str.trim(@model.grProperties.propVal('link', 'label'))
      label = "Go to '#{link}'" if _.str.isBlank(label)

      target = @model.grProperties.propVal('link', 'target')
      if _.str.isBlank(target)
        openInNewWindow = _.str.toBoolean(@model.grProperties.propVal('link', 'openInNewWindow'))
        target = openInNewWindow and '_blank' or '_self'

      href = link
      href = "?grname=#{link}" if isGraphicLink(link)

      if target == "NewWindow"
        $("""
        <li><a href='#' onclick='var newWin = open("#{href}", "_blank", "width=400, height=300"); newWin.moveTo(200, 200); return false;'>#{label}</a></li>
        """).appendTo(menuElem)
      else
        $("<li><a href='#{href}' target='#{target}'>#{label}</a></li>").appendTo(menuElem)

      # if _.str.startsWith(link, 'http')
      #   $("<li><a href='#{link}'>Go to '#{link}'</a></li>").appendTo(menuElem)
      # else if _.str.endsWith(link, '.gr')
      #   $("<li><a href='?grname=#{link}'>Go to '#{link}'</a></li>").appendTo(menuElem)
      # else
      #   console.error "invalid link value: " + link
      #   return

      $("<li class='divider'></li>").appendTo(menuElem)

    render: ()->
      @$el.empty()

      return unless @model.grMenuActions.hasVisibleAction()

      ul = $("<ul class='dropdown-menu' role='menu'></ul>").appendTo(@$el)
      @renderLinkMenu(ul)
      for menuAction in @model.grMenuActions.models when menuAction.get('show') == '1'
        @renderCompMenuActions(menuAction, ul)
      @

  class GrObjectView extends Backbone.View
    className: 'grObject'
    contextMenu: new GrMenuView
    askParameterDialog: new GrAskParameterDialog

    initialize: ->
      properties = @model.grProperties
      @listenTo(properties, 'change', @applyPropChanged)

      visibleProp = properties.get('visible')
      @listenTo(visibleProp, 'change:value', @applyVisibility) if visibleProp

      imageProp = properties.get('image')
      @listenTo(imageProp, 'change:value', @applyImage) if imageProp

      rectProp = properties.get('rect')
      @listenTo(rectProp, 'change:value', @applyRect) if rectProp

      bgColorProp = properties.get('backgroundColor')
      @listenTo(bgColorProp, 'change:value', @applyBackgroundColor) if bgColorProp

      @listenTo(properties, 'dataOutOfBound:image', @applyImageDataOutOfBound)

      @initMenu()

      @setCursorUpdater()

    hasWritePerm: ->
      window.grCanvasView.model.actionPermitted

    clickable: ->
      @model.grMenuActions.hasVisibleAction() or (not _.str.isBlank(@model.grProperties.propVal("link")))

    setCursorUpdater: ->
      return unless @clickable()

      @$el.mouseenter =>
          @$el.css('cursor', 'pointer')
      @$el.mouseleave =>
          @$el.css('cursor', 'initial')

    initMenu: ->
      unless @model.dataSource()? or @model.grMenuActions.hasVisibleAction()
        @$el.on('click', @onClick)
        return

      unless @hasWritePerm()
        @$el.on('click', @onClick)
        return

      @$el.contextmenu({
        target: "#grObjectMenu",
        before: (e, element, target)=>
          @contextMenu.model = @model
          unless @contextMenu.render()
            @onClick()
            return false
          e.preventDefault()
          true

        onItem: (e, item)=>
          menuItemData = item.data("menuItemData")
          @askParameterDialog.menuItemData = menuItemData
          @askParameterDialog.render()
          true
      })

      @frames = 0
      @curFrameIndex = 0
      @animationTimerId = null

    bindPropNames: ->
      ['visible', 'image', 'rect', 'backgroundColor']

    # the property 'change:value' event may triggered before view rendered,
    # that will cause event handler not triggered, call this method to
    # trigger the 'change:value' event manually
    triggerBindPropEvents: =>
      properties = @model.grProperties
      return unless properties
      for name in _.uniq(@bindPropNames())
        continue unless prop = properties.get(name)
        continue unless val = properties.propVal(name)
        prop.trigger('change:value', @model, val)

    render: ->
      @applyRectInfo(@$el, "rect")

      zorder = @model.grProperties.propVal("zorder")
      @$el.css('z-index', zorder) if !!zorder

      @applyBackgroundColorInfo(@model.grProperties.propVal("backgroundColor"))

      @renderImage()

      _.delay(@triggerBindPropEvents, 100)

      @$el.hide() unless @model.isVisible()

      @

    renderImage: ->
      imagePath = @model.grProperties.propVal("image")
      if !!imagePath
        @buildImageItem(imagePath)

    buildImageItem: (imagePath)->
      imgWrapper = $("<div class='grImage'/>").appendTo(@$el)
      @applyRectInfo(imgWrapper, 'image', '_rect')
      if @hasAnimation(imagePath)
        imgWrapper.css "background-image", "url('#{imagePath}')"
      else
        $("<img src='#{imagePath}'>").appendTo(imgWrapper)

    applyImage: (model, value, options) =>
      if @hasAnimation(value)
        @$('.grImage').css('background-image', "url('#{value}')") if !!value
      else
        @$('.grImage img').attr('src', value) if !!value

      @stopAnimation()
      if @hasAnimation(value)
        @applyAnimation(value)

    hasAnimation: (value)->
      return /\/_pngAnimations\//.test(value)

    applyAnimation: (value)->
      pat = /^.*_f(\d+).png$/
      result = pat.exec(value)
      unless result and result.length >= 2
        return

      @frames = _.str.toNumber(result[1])
      @curFrameIndex = 0
      @animationTimerId = setInterval(@animateImage, 100)

    animateImage: =>
      elem = @$('.grImage')
      width = elem.css('width')
      width = _.str.strLeft(width, "px") if _.str.endsWith(width, "px")
      width = _.str.toNumber(width)
      position = "#{width*@curFrameIndex}px 0px"
      elem.css('background-position', position)
      @curFrameIndex = (@curFrameIndex+1) % @frames

    stopAnimation: ->
      clearInterval(@animationTimerId) if @animationTimerId

    applyRect: (model, value, options) =>
      @applyRectInfo(@$el, "rect")

    applyRectInfo: (object, rectPropName, propAttrName)->
      attrs = ['left', 'top', 'width', 'height']
      rectValues = @model.grProperties.propVal(rectPropName, propAttrName)?.split(',')
      return unless rectValues?
      rectValues = _.map(rectValues, (v)->
        "#{v}px"
      )
      object.css(_.object(attrs, rectValues))

    applyBackgroundColor: (model, value, option)=>
      @applyBackgroundColorInfo(value)

    applyImageDataOutOfBound: (name, value)=>
      @$el.fadeOut()

    unpackColorData: (colorData)->
      return unless !!colorData
      parts = _.str.words(colorData, ',')
      return unless parts.length == 3 or parts.length == 4
      # NOTE: rgba does not support IE before version 10
      if parts.length == 4
        parts[3] = _.str.toNumber(parts[3])/256.0
      rgba = parts.join(',')

    applyBackgroundColorInfo: (colorData)->
      rgba = @unpackColorData(colorData)
      return unless rgba?
      @$el.css("background-color", "rgba(#{rgba})")

    applyPropChanged: =>
      # restore widget's visibility when prop changed, the widget maybe hiden
      # because of data out of bound
      return unless @model.isVisible()
      @$el.fadeIn() unless @$el.is(":visible")

    applyVisibility: (name, value)=>
      if _.str.toBoolean(value)
        @$el.fadeIn()
      else
        @$el.fadeOut()

    onClick: =>
      link = @model.grProperties.propVal("link")
      return if _.str.isBlank(link)

      if isGraphicLink(link)
        queryStr = location.search
        if _.str.isBlank(queryStr)
            queryStr = "?grname=#{link}"
        else
            paramPat = /([&?])grname=[^&#=]*/
            if paramPat.test(queryStr)
                queryStr = queryStr.replace(paramPat, "$1grname=#{link}")
            else
                queryStr = queryStr + "&grname=#{link}"

        link = location.origin + location.pathname + queryStr + location.hash

      target = @model.grProperties.propVal('link', 'target')
      if _.str.isBlank(target)
          openInNewWindow = _.str.toBoolean(@model.grProperties.propVal("link", "openInNewWindow"))
          target = openInNewWindow and '_blank' or '_self'

      if target != "_self"
          if target == "NewWindow"
              newWin = window.open(link, "_blank", "width=400, height=300")
              newWin.moveTo(200, 200)
          else
              window.open(link, target)
      else
          redirect(link)

  class GrLabelView extends GrObjectView
    initialize: ->
      super

      properties = @model.grProperties

      textProp = properties.get('text')
      @listenTo(textProp, 'change:value', @applyText) if textProp

      fontProp = properties.get('font')
      @listenTo(fontProp, 'change:value', @applyFont) if fontProp

      textColorProp = properties.get('textColor')
      @listenTo(textColorProp, 'change:value', @applyTextColor) if textColorProp

    bindPropNames: ->
      _.union(super, ['text', 'font', 'textColor'])
      
    getLabelText: ->
      @model.grProperties.propVal("text")

    render: ->
      super
      @$el.css("overflow", "visible")

      grProperties = @model.grProperties
      text = @getLabelText()
      if !!text
        # txtElem = $("<div class='grText' >#{text}</div>").appendTo(@$el)
        txtElem = $("<div class='grText' ></div>").appendTo(@$el)
        @applyRectInfo(txtElem, 'text', '_rect')
        TextFormatter.applyFontInfo(txtElem, @model.grProperties.propVal("font"))
        TextFormatter.applyTextColorInfo(txtElem, grProperties.propVal("textColor"))
        @applyText(null, text, null)
      @

    isPureTextLabel: ->
      @$el.find('.grImage').length == 0

    applyText: (model, value, options) =>
      if !!value
        @$('.grText').text(value)
      else
        @$('.grText').text('')
      @applyAlignment() if @isPureTextLabel()

    applyAlignment: (txtElem)->
      txtElem = @$el.find('.grText') unless txtElem
      grProperties = @model.grProperties
      hAlign = _.str.toNumber(grProperties.propVal('hAlign'))
      [leftAlign, centerAlign, rightAlign] = [0, 1, 2]
      if hAlign == rightAlign
        txtElem.css('text-align', 'right')

        # adjust 'left' and 'width' to have enough space for text
        # if element's width is smaller than text's width, text will always be
        # displayed as left aligned
        textWidth = txtElem.textWidth(txtElem.text())
        elemWidth = txtElem.width()
        if textWidth > elemWidth
          left = txtElem.css('left')
          left = _.str.toNumber(left.substring(0, left.length-2))
          txtElem.css(
            'left': left-(textWidth+4-elemWidth)
            'width': textWidth+4
          )
      else if hAlign == leftAlign
        txtElem.css('text-align', 'left')
      else if hAlign == centerAlign
        txtElem.css('text-align', 'center')

    applyFont: (model, value, options) =>
      font = @model.grProperties.propVal("font")
      TextFormatter.applyFontInfo @$('.grText'), font

    applyTextColor: (model, value, option)=>
      TextFormatter.applyTextColorInfo(@$el.find('.grText'), value)

  class GrWebLineEditView extends GrObjectView

    initialize: ->
      super
      properties = @model.grProperties

      inputDataProp = properties.get('inputData')
      @listenTo(inputDataProp, 'change:value', @applyInputData) if inputDataProp
      @btnsWidth = 130
      @origZIndex = 0
      @okToSubmit = true
      @inputType = "text"

      @inputElemClass = 'grWebLineEdit'

      @listenTo(@, 'input:valid', @enableSubmit)
      @listenTo(@, 'input:invalid', @disableSubmit)

    initMenu: ->
      # disable contextual menu

    bindPropNames: ->
      _.union(super, ['inputData'])

    renderImage: ->
      @

    createInputElem: (slotType, formElem)->
      rangeLabels = slotType?.rangeLabels()
      unitLabel = slotType?.get('unitLabel')
      if not _.isEmpty(rangeLabels)
        options = _.str.toSentence("<option value='#{index}'>#{label}</option>" for label, index in rangeLabels)
        inputElem = $("<select data-isrange='1' class='#{@inputElemClass}'>#{options}</select>").appendTo(formElem)
      else if slotType.isBoolean()
        options = """
          <option value='false'>false</option>
          <option value='true'>true</option>
        """
        # GOTCHA: 'nil' represent 'null' value here, because 'null' text in
        # param will crash appweb due to appweb's bug 
        options += "<option value='nil'>null</option>" if slotType.isAllowNull()

        inputElem = $("""
        <select class='#{@inputElemClass}'>
          #{options}
        </select>
        """).appendTo(formElem)
      else
        inputElem = $("<input class='#{@inputElemClass}' type='#{@inputType}'>").appendTo(formElem)

      inputElem.css("width", @$el.width()-14)
      inputElem

    renderInputElem: (formElem)->
      vevent = @model.grValueEvents.get('inputData')
      return unless vevent?
      dataSource = vevent.grDataSource
      typeInfo = dataSource.compTypeInfo()
      slotType = typeInfo.getProperty(dataSource.slotName())

      inputElem = @createInputElem(slotType, formElem)
      inputElem.focusin(@onFocusIn)
      inputElem

    render: ->
      super
      @$el.css("overflow", "visible")

      formElem = $("<form class='form-inline'></form>").appendTo(@$el)
      @renderInputElem(formElem)
      @renderButtons(formElem)
      formElem.submit(@submitHandler)

    renderButtons: (formElem)->
      okBtn = $("<button type='submit' class='btn btn-primary hide'>Ok</button>").appendTo(formElem)
      closeBtn = $("<button type='button' class='btn hide'>Close</button>").appendTo(formElem)
      okBtn.css('margin-left': 4)
      closeBtn.css('margin-left': 2)
      closeBtn.click(@onCloseEditor)

    showButtons: =>
      return unless @hasWritePerm()
      return if @$el.find("button").is(":visible")
      @origZIndex = @$el.css('z-index')
      @$el.css('z-index', maxZIndex()+1)
      @$el.width(@$el.width()+@btnsWidth).find("button").fadeIn(400)

      elem = @$(".#{@inputElemClass}")
      if (elem.is('select'))
        @prevOptionVal = elem.val()

    onFocusIn: =>
      @showButtons()
      
    preprocessVal: (val)=>
      val

    applyInputData: (model, value, options) =>
      @applyData(@preprocessVal(value))

    applyData: (value) =>
      elem = @$(".#{@inputElemClass}")
      return if !elem
      if elem.is('input')
        elem.attr("placeholder", value)
        elem.val(value)
      else if elem.is('select')
        if elem.data('isrange')
          elem.find('option').filter(->
            $(this).text() == value
          ).prop('selected', true)
        else
          elem.find('option').filter(->
            val = $(this).val()
            if value == 'null'
              # GOTCHA: 'null' value should match option with 'nil' value, this
              # workaround because of appweb's bug. should remove it when appweb's bug fixed
              val == value or val == 'nil'
            else
              val == value
          ).prop('selected', true)

    submitHandler: (event)=>
      event.preventDefault()
      return unless @okToSubmit
      @apply()

    value: ->
      @$el.find(".#{@inputElemClass}").val()

    apply: ->
      return unless @hasWritePerm()

      compPool = ComponentsPool.instance()
      vevent = @model.grValueEvents.get('inputData')
      dataSource = vevent.grDataSource
      comp = dataSource.component()
      typeInfo = dataSource.compTypeInfo()
      slotType = typeInfo.getProperty(dataSource.slotName())
      url = compPool.urlFor(comp, slotType.get('name'))

      # TODO: move following logics into SlotType class, add a 'getType' method
      slotDataType = slotType.get('type')
      slotDataType = 'Buf' if slotDataType == 'str'

      value = @value()
      # GOTCHA: fix appweb's crash bug to handle 'null' param 
      value = 'nil' if slotType.isFloatNumber() and value == 'null'

      $.ajax
        type: "POST"
        url: compPool.proxyUrl()
        data:
          path: url
          type: slotDataType
          value: value
          slotType: 'property'
        dataType: 'json'
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled")
          @$el.find('.btn-primary').attr("disabled", "disabled")
        success: (data)=>
          if data.error
            console.error("failed to change data slot: #{data.error.text}")
          else if data.redirect?
            redirect(data.redirect)
          else
            @closeEditor()
            @applyData(@preprocessVal(@value()))
            #@$el.find('form')[0].reset()
        complete: (data, status)=>
          @$el.find('.btn-primary').removeClass("disabled")
          @$el.find('.btn-primary').removeAttr("disabled")

    onCloseEditor: =>
      @closeEditor()
      # restore input's value to last updated value, that is stored as placeholder
      elem = @$(".#{@inputElemClass}")
      if elem.is('input')
        elem.val(elem.attr("placeholder"))
      else if elem.is('select')
        if @prevOptionVal
          elem.val(@prevOptionVal)
        else
          elem.find('option').prop('selected', ->
            this.defaultSelected
          )
      else
        console.warn("unknown element type, try to restore its value")
        elem.val(elem.attr("placeholder"))

    closeEditor: =>
      # GOTCHA: $.when is required because there are two buttons, the restore-width callback will be called twice, but with $.when and 'done'
      $.when(@$el.find("button").fadeOut(400)).done(
          =>
              @$el.width(@$el.width()-@btnsWidth)
              @$el.css('z-index', @origZIndex)
      )
      @$(".#{@inputElemClass}").blur()

    enableSubmit: =>
      @okToSubmit = true
      @$el.find('button.btn-primary').removeClass('disabled')

    disableSubmit: =>
      @okToSubmit = false
      @$el.find('button.btn-primary').addClass('disabled')
      
  class GrWebPasswordEditView extends GrWebLineEditView
    initialize: ->
      super
      @inputType = "password"

  class GrWebLineEditWithValidationView extends GrWebLineEditView
    initialize: ->
      super
      @origInputValue = ''

    renderInputElem: (formElem)->
      inputElem = super(formElem)
      return unless inputElem?
      inputElem.keypress @onKeyPress
      inputElem.on "input paste", @onPasted

    onFocusIn: =>
      super()
      @validateInput()

    onPasted: =>
      @validateInput()

    isNonPrintableChar: (event)->
      if (event.which >= 32 and event.which <= 126)
        return (event.metaKey || event.altKey || event.ctrlKey)
      else
        return true

    onKeyPress: (event)=>
      return if @isNonPrintableChar(event)

      @origInputValue = @$el.find(".#{@inputElemClass}").val()
      setTimeout(@validateInput, 0)

  class GrWebIPAddressEditView extends GrWebLineEditWithValidationView

    validateInput: =>
      val = @$el.find(".#{@inputElemClass}").val()
      return if _.str.isBlank(val)

      val = _.str.trim(val)
      inputElem = @$el.find(".#{@inputElemClass}")

      pat = /^[1-9][0-9]{0,2}(\.[0-9]{1,3}){0,3}$/
      m = pat.test(val)
      if not m
        m = pat.test(val.substring(0, val.length-1))
      unless m
        inputElem.val(@origInputValue)
        return

      parts = val.split('.')
      for p, i in parts
        if _.str.isBlank(p)
          continue
        if p.length > 3
          p = p.substring(0, 3)
        iVal = _.str.toNumber(p)
        iVal = Math.min(255, iVal)
        parts[i] = iVal.toString()
      validIP = parts.slice(0, Math.min(4, parts.length)).join('.')

      inputElem.val(validIP)

  class GrWebTimeRangeEditView extends GrWebLineEditWithValidationView

    validateInput: =>
      val = @$el.find(".#{@inputElemClass}").val()

      # when input is empty, it means delete the item completely
      # maybe we should add a 'allowEmpty' property
      if _.str.isBlank(val)
        @trigger('input:valid')
        return

      val = _.str.trim(val)
      inputElem = @$el.find(".#{@inputElemClass}")

      pat = /^((2[0123])|(1[0-9])|(0[0-9]))([0-5][0-9])-((2[0123])|(1[0-9])|(0[0-9]))([0-5][0-9])$/
      if pat.test(val)
        @trigger('input:valid')
      else
        @trigger('input:invalid')

  class GrWebDateRangeEditView extends GrWebLineEditWithValidationView

    validateInput: =>
      val = @$el.find(".#{@inputElemClass}").val()
      
      # refer above comments
      if _.str.isBlank(val)
        @trigger('input:valid')
        return

      val = _.str.trim(val)
      inputElem = @$el.find(".#{@inputElemClass}")

      if _.str.endsWith(val, '-')
        @trigger('input:invalid')
        return

      # format: dd/mm/yy-dd/mm/yy, or **/**/**
      parts = _.map(_.str.words(val, '-'), (w)->
        _.str.trim(w)
      )

      if parts.length != 1 and parts.length != 2
        @trigger('input:invalid')
        return

      properties = @model.grProperties
      dateFormat = properties.propVal("dateFormat")
      dateFormat ?= "0"

      unless @isDateValid(parts[0], dateFormat)
        @trigger('input:invalid')
        return

      if parts.length == 2
        unless @isDateValid(parts[1], dateFormat)
          @trigger('input:invalid')
          return

      @trigger('input:valid')

    isDateValid: (dateStr, dateFormat)->
      return false if _.str.endsWith(dateStr, '/')
      parts = _.str.words(dateStr, '/')
      return false unless parts.length == 3
      return false if _.some(parts, (p)->
        p.length != 2
      )
      if (dateFormat == "0")
        [dd, mm, yy] = parts
      else if (dateFormat == "1")
        [mm, dd, yy] = parts
      else
        return false
      idd = if dd == '**' then 1 else _.str.toNumber(dd)
      imm = if mm == '**' then 0 else _.str.toNumber(mm)-1
      iyy = if yy == '**' then 2000 else 2000+_.str.toNumber(yy)
      d = new Date(iyy, imm, idd)
      d.getFullYear() == iyy and d.getMonth() == imm and d.getDate() == idd

  class GrWebNumberRangeEditView extends GrWebLineEditWithValidationView

    initialize: ->
      super
      properties = @model.grProperties
      @max = parseFloat(properties.propVal('max'))
      @min = parseFloat(properties.propVal('min'))
      
    value: ->
      val = super
      parseFloat(val)

    validateInput: =>
      val = @$el.find(".#{@inputElemClass}").val()
      if @isDataValid(val)
        @trigger('input:valid')
      else
        @trigger('input:invalid')

    isDataValid: (valStr)->
      numVal = parseFloat(valStr)
      if _.isNaN(numVal)
        return false

      unless _.isNaN(@max)
        return false if numVal > @max

      unless _.isNaN(@min)
        return false if numVal < @min

      return true

  class GrWebHexNumberEditView extends GrWebLineEditWithValidationView
    initialize: ->
      super
      properties = @model.grProperties
      @max = parseInt(properties.propVal('max'))
      @min = parseInt(properties.propVal('min'))
      
    value: ->
      val = super
      val = '0x'+val unless _.str.startsWith(val, '0x')
      val

    validateInput: =>
      if @isDataValid(@value())
        @trigger('input:valid')
      else
        @trigger('input:invalid')

    isDataValid: (valStr)->
      valStr = '0x'+valStr unless _.str.startsWith(valStr, '0x')
      return false unless /^0x[0-9a-fA-F]+$/.test(valStr)

      numVal = parseInt(valStr)
      if _.isNaN(numVal)
        return false

      unless _.isNaN(@max)
        return false if numVal > @max

      unless _.isNaN(@min)
        return false if numVal < @min

      return true

  class GrWebTimeEditView extends GrWebLineEditWithValidationView

    validateInput: =>
      val = @$el.find(".#{@inputElemClass}").val()

      # when input is empty, it means delete the item completely
      # maybe we should add a 'allowEmpty' property
      if _.str.isBlank(val)
        @trigger('input:valid')
        return

      val = _.str.trim(val)
      inputElem = @$el.find(".#{@inputElemClass}")

      pat = /^((2[0123])|(1[0-9])|(0[0-9]))([0-5][0-9])$/
      if pat.test(val)
        @trigger('input:valid')
      else
        @trigger('input:invalid')

  class GrWebSliderEditView extends GrWebLineEditView

    initialize: ->
      super
      @inputElemClass = 'grSliderEdit'

    createInputElem: (slotType, formElem)->
      properties = @model.grProperties
      max = _.str.toNumber(properties.propVal('max') ? '100')
      min = _.str.toNumber(properties.propVal('min') ? '0')

      prefix = $("<span>#{min}</span>").appendTo(formElem)
      inputElem = $("<input class='#{@inputElemClass}' type='text'>").appendTo(formElem)
      suffix = $("<span>#{max}</span>").appendTo(formElem)

      inputElem.slider({max: max, min: min}).on('slideStart', @showButtons)

      # tweak elements size and position
      margin = @$el.find('.slider-handle').outerWidth()/2 + 1
      width = @$el.width()-@$el.textWidth("#{min}")-@$el.textWidth("#{max}")-margin*2 - 2
      @$el.find('.slider').css({'width': width, 'margin-left': margin, 'margin-right': margin})

      # GOTCHA: update slider's internal size
      #         the size is set during slider's constructor, even we
      #         update its width, but the size will not change, so hack it here
      inputElem.data('slider')['size'] = width

      # GOTCHA: hack tooltip's top position, the computed value is not good enough
      @$el.find('.tooltip.top').css('top', '+=-20')

      inputElem

    applyData: (value) =>
      elem = @$(".#{@inputElemClass}")
      return if !elem or elem.is(':focus')

      elem.slider('setValue', value)

  class GrWebDateTimeEditView extends GrWebLineEditView

    initialize: ->
      super
      @inputElemClass = 'grDateTimeEdit'
      @btnsWidth = 120

    createInputElem: (slotType, formElem)->
      inputElem = $("<input data-format='dd/MM/yyyy hh:mm:ss' type='text'></input>").appendTo(formElem)
      inputElem.css('width', @$el.width()-24)
      $("<span class='add-on'><i data-time-icon='icon-time' data-date-icon='icon-calendar'> </i></span>").appendTo(formElem)
      @$el.addClass("input-append").addClass("date")
      @$el.datetimepicker({language: 'pt-BR'})
      @$el.on("changeDate", @onDateTimeChanged)
      inputElem

    onDateTimeChanged: =>
      @showButtons()

  class GrLineChartView extends GrObjectView

    initialize: ->
      super

      @plot = null
      @buildLineDatas()

    buildLineDatas: ->
      properties = @model.grProperties
      @showLastNMin = _.str.toNumber(properties.propVal('showLastNMin') ? '20')

      @lineDatas = new LineDatas
      @lineDatas.propertySet = properties
      @lineDatas.keepDataOfLastNMin = @showLastNMin
      for i in [1..4]
        name = "line#{i}"
        continue unless @model.isPropertyBound(name)

        options =
          name: name
          label: @labelFor(i)
          data: []
          lastUpdateTimeMs: null
        color = properties.propVal("colorOfLine#{i}")
        options["color"] = "rgba(#{color})" unless _.str.isBlank(color)
        @lineDatas.add(options)

      @listenTo(ComponentsPool.instance(), 'sync', @onDataSynced)

    labelFor: (index)->
      @model.grProperties.propVal("labelOfLine#{index}") ? "line#{index}"

    renderImage: ->
      @

    toolTipLabel: (item)->
      y = item.datapoint[1].toFixed(2)
      "#{y} of #{item.series.label}"

    renderToolTip: ->
      $("<div id='tooltip'></div>").css({
        position: "absolute",
        display: "none",
        padding: "2px",
        "background-color": "#fefefe",
        opacity: 0.85,
      }).appendTo("body")

      @$el.on("plothover", (event, pos, item)=>
        zIndex = _.str.toNumber(@$el.css('z-index')) + 1
        if (item)
          $("#tooltip").html(@toolTipLabel(item))
          .css({
            top: item.pageY+5
            left: item.pageX+5
            'z-index': 100
            'border': "2px solid #{item.series.color}"
          })
          .fadeIn(200)
        else
          $("#tooltip").hide()
      )

    render: ->
      super
      @$el.css("overflow", "visible")

      plotContainer = $('<div class="plotContainer pull-left" />').appendTo(@$el)
      plotContainer.height(@$el.height())
      plotContainer.width(@$el.width()-60)

      legendContainer = $('<div class="legendContainer pull-left"/>').appendTo(@$el)
      legendContainer.height(@$el.height())
      legendContainer.width(60)

      @renderToolTip()
      setTimeout(@buildChart, 0)

    tickFormatter: (val, axis)->
      return 'now' if val == 0

      val *= -1 if val < 0
      index = 0
      levels = [1000, 60*1000, 60*60*1000, 24*60*60*1000]
      for l in levels
        if val >= l
          ++index
        else
          break

      index = Math.max(0, --index)
      units = ['s', 'm', 'h', 'd']
      valInUnit = val/levels[index]
      if val%levels[index] > 0
        valInUnit = valInUnit.toFixed(2)
      "#{valInUnit}#{units[index]}"

    buildChart: =>
      properties = @model.grProperties

      @plot = $.plot(@$el.find('.plotContainer'), @lineDatas.datas(),
        series:
          shadowSize: 0
          points:
            show: false
          lines:
            show: true
        xaxis:
          show: true
          # must have min and max set
          min: -1*@showLastNMin*60*1000
          max: 0
          mode: 'time'
          tickFormatter: @tickFormatter
          tickDecimals: 0
          minTickSize: [10, "second"]
        yaxis:
          min: _.str.toNumber(properties.propVal('minOfYAxis') ? '0.0')
          max: _.str.toNumber(properties.propVal('maxOfYAxis') ? '100.0')
          tickDecimals: 2
        grid:
          hoverable: true
        legend:
          container: @$el.find('.legendContainer')
      )

      @refreshChart()

    onDataSynced: ->
      compsPool = ComponentsPool.instance()
      return unless compsPool.isBatchReqDone()

      @refreshChart()

    refreshChart: =>
      return unless @plot
      @lineDatas.updateDatas()
      @plot.setData(@lineDatas.datas())
      @plot.draw()
      
  class GrDateTimePicker

    initialize: ->
      @elem = null
      @dateTimeRanges = null
      @startDate = moment().startOf('day')
      @endDate = moment().add('days', 1).startOf('day')
      
    attach: (elem, weekType='isoWeek', valueChanged=null)->
      @elem = elem
      todayStr = L("Today")
      yesterdayStr = L("Yesterday")
      last24HrStr = L('Last 24hr')
      weekToDateStr = L('Week to date')
      lastWeekStr = L('Last week')
      monthToDateStr = L('Month to date')
      lastMonthStr = L('Last Month')
      yearToDateStr = L('Year to date')
      lastYearStr = L('Last year')
      @dateTimeRanges = {}
      @dateTimeRanges[todayStr] = [moment().startOf('day'), moment().add('days', 1).startOf('day')]
      @dateTimeRanges[yesterdayStr] = [moment().subtract('days', 1).startOf('day'), moment().startOf('day')]
      # 'Test':          [moment("2014-01-22 18:00:00"), moment("2014-01-22 18:01:00")]
      @dateTimeRanges[last24HrStr] = [moment().subtract('hours', 24), moment()]
      @dateTimeRanges[weekToDateStr] = [moment().startOf(weekType), moment().add('days', 1).startOf('day')]
      @dateTimeRanges[lastWeekStr] = [moment().subtract('weeks', 1).startOf(weekType), moment().startOf(weekType)]
      @dateTimeRanges[monthToDateStr] = [moment().startOf('month'), moment().add('days', 1).startOf('day')]
      @dateTimeRanges[lastMonthStr] = [moment().subtract('months', 1).startOf('month'), moment().startOf('month')]
      @dateTimeRanges[yearToDateStr] = [moment().startOf('year'), moment().add('days', 1).startOf('day')]
      @dateTimeRanges[lastYearStr] = [moment().subtract('years', 1).startOf('year'), moment().startOf('year')]

      todayRange = @dateTimeRanges[todayStr]
      @startDate = todayRange[0]
      @endDate = todayRange[1]
      $(elem).daterangepicker(
        startDate: @startDate
        endDate: @endDate
        timePicker: true
        timePicker12Hour: false
        timePickerIncrement: 1
        showDropdowns: true
        format: 'MM/DD/YYYY HH:mm'
        ranges: @dateTimeRanges
        , valueChanged)

    rangeLabel: (start, end)->
      # start.format('YYYY/MM/DD HH:mm') + ' - ' + end.format('YYYY/MM/DD HH:mm')
      invertedRanges = _.invert(@dateTimeRanges)
      if _.has(invertedRanges, [start, end])
        invertedRanges[[start, end]]
      else
        start.format('YYYY/MM/DD HH:mm') + ' - ' + end.format('YYYY/MM/DD HH:mm')
        
    update: ->
      picker = $(@elem).find('.datetime-range').data('daterangepicker')
      return unless picker
      picker.setStartDate(@startDate)
      picker.setEndDate(@endDate)
      
    updateRange: (start, end)->
      @startDate = start
      @endDate = end
      $(@elem).find(".rangeLabel").text(@rangeLabel(start, end))

  class GrHistoryLineChartView extends GrLineChartView

    initialize: ->
      super

      @spinner = null
      @plot = null

      @dateTimePicker = new GrDateTimePicker()
      @dateTimePicker.initialize()
      @listenTo(SystemEventSingleton.instance(), 'weekDayChanged', @refreshDateTimePicker)
      
    refreshDateTimePicker: =>
      console.debug("refresh datetime picker")
      @attachDateTimePicker(@$el.find(".datetime-range"))
      
    attachDateTimePicker: (elem)->
      properties = @model.grProperties
      if properties.propVal("weekStartDay") == '1' # start from Sunday
        weekType = 'week'
      else
        weekType = 'isoWeek'
      
      @dateTimePicker.attach(elem, weekType, @onDateTimeRangeChanged)

    buildLineDatas: ->
      properties = @model.grProperties

      @lineDatas = new HistoryLineDatas
      @lineDatas.propertySet = properties
      for i in [1..4]
        name = properties.propVal("colNameOfLine#{i}")
        continue if _.str.isBlank(name)

        options =
          name: name
          label: @labelFor(i)
          data: []
        color = properties.propVal("colorOfLine#{i}")
        options["color"] = "rgba(#{color})" unless _.str.isBlank(color)
        @lineDatas.add(options)

    render: ->
      super

      @$el.find('.plotContainer').addClass('rotateTicks')

      label = L('Today')
      html = """
        <div class="datetime-range pull-right clearfix" style="margin-right: 66px; margin-bottom: 4px;">
          <button class="btn" type="button">
          <i class="icon-calendar"></i> <span class="rangeLabel">#{label}</span> <span class="caret"></span>
          </button>
        </div>
      """
      @attachDateTimePicker($(html).prependTo(@$el))

    buildChart: =>
      properties = @model.grProperties

      @plot = $.plot(@$el.find('.plotContainer'), @lineDatas.datas(),
        series:
          shadowSize: 0
          points:
            show: false
          lines:
            show: true
        xaxis:
          show: true
          mode: 'time'
          minTickSize: [10, "second"]
          # timeFormat: "%Y/%m/%d %H:%M:%S"
          tickFormatter: @tickFormatter
          min: @dateTimePicker.startDate.unix()*1000
          max: @dateTimePicker.endDate.unix()*1000
        yaxis:
          tickDecimals: 2
        grid:
          hoverable: true
        legend:
          container: @$el.find('.legendContainer')
        selection:
          mode: "x"
      )

      @listenTo(@lineDatas, 'sync', @refreshChart)

      @$el.find('.plotContainer').bind("plotselected", @onSelectionChanged)

      @onDateTimeRangeChanged(@dateTimePicker.startDate, @dateTimePicker.endDate)

    toolTipLabel: (item)->
      y = item.datapoint[1].toFixed(2)
      x = moment(item.datapoint[0]).format('YYYY-MM-DD HH:mm')
      # "#{item.series.label}: #{y} at #{x}"
      "#{y} at #{x}"

    tickFormatter: (val, axis)=>
      moment(val).format('YYYY-MM-DD HH:mm')

    updateDynamicDate: ->
      if L('Last 24hr') == @$el.find(".datetime-range .rangeLabel").text()
        @dateTimePicker.startDate = moment().subtract('hours', 24)
        @dateTimePicker.endDate = moment()

    updateTimeRange: (start, end)->
      @dateTimePicker.startDate = start
      @dateTimePicker.endDate = end

      @$el.find(".datetime-range .rangeLabel").text(@dateTimePicker.rangeLabel(start, end))

    onDateTimeRangeChanged: (start, end)=>
      @updateTimeRange(start, end)
      @updateData()

    onSelectionChanged: (event, ranges)=>
      options = @plot.getOptions()
      xaxis = options.xaxes[0]
      xaxis.min = ranges.xaxis.from
      xaxis.max = ranges.xaxis.to

      @plot.setupGrid()
      @plot.draw()
      @plot.clearSelection()

      @updateTimeRange(moment(xaxis.min), moment(xaxis.max))

      # update timepicker dropdown menu so that it is synced with selection
      # time range
      @dateTimePicker.update()

    updateData: ->
      @updateDynamicDate()

      @spinner = new Spinner() unless @spinner
      @spinner.spin(@$el[0])

      @lineDatas.updateDatas(@dateTimePicker.startDate, @dateTimePicker.endDate)

    refreshChart: ->
      @spinner.stop()
      return unless @plot

      @plot.setData(@lineDatas.datas())

      options = @plot.getOptions()
      options.xaxes[0].min = @dateTimePicker.startDate.unix()*1000
      options.xaxes[0].max = @dateTimePicker.endDate.unix()*1000

      @plot.setupGrid()
      @plot.draw()

  class GrTableView extends GrObjectView
    initialize: ->
      super

      properties = @model.grProperties

      @showLastNMin = _.str.toNumber(properties.propVal('showLastNMin') ? '20')

      unless _.str.isBlank(properties.propVal('minAlertThreshold'))
        @minAlertThreshold = _.str.toNumber(properties.propVal('minAlertThreshold'))
      unless _.str.isBlank(properties.propVal('maxAlertThreshold'))
        @maxAlertThreshold = _.str.toNumber(properties.propVal('maxAlertThreshold'))

      @lineDatas = new LineDatas
      @lineDatas.propertySet = properties
      @lineDatas.keepDataOfLastNMin = @showLastNMin
      for i in [1..4]
        name = "column#{i}"
        continue unless @model.isPropertyBound(name)

        options =
          name: name
          label: @labelFor(i)
          data: []
          lastUpdateTimeMs: null
        @lineDatas.add(options)

      @listenTo(ComponentsPool.instance(), 'sync', @onDataSynced)

    labelFor: (index)->
      @model.grProperties.propVal("labelOfColumn#{index}") ? "column#{index}"

    renderImage: ->
      @

    render: ->
      super
      @$el.css("overflow", "auto")

      table = _.template($('#table_view_tpl').html(), {columns: @lineDatas.models})
      @$el.append(table)

      @refreshTable()

    onDataSynced: ->
      compsPool = ComponentsPool.instance()
      return unless compsPool.isBatchReqDone()

      @refreshTable()

    nowTimeLabel: ->
      now = new Date()
      parts = [now.getHours(), now.getMinutes(), now.getSeconds()]
      _.map(parts, (p)->
        _.str.pad(p, 2, '0')
      ).join(':')

    refreshTable: ->
      @lineDatas.updateDatas()
      columns = _.map(@lineDatas.models, (m)->
        m.latestVal()
      )

      isValid = _.every(columns, (c)->
        not _.isNaN(c) and not _.isNull(c) and not _.isUndefined(c) and _.isNumber(c)
      )
      return unless isValid

      tbody = @$el.find("table > tbody")
      html = _.template("""
      <tr>
        <td><%= now %></td>
        <% _.each(columns, function(col) { %>
        <td><%= col %></td>
        <% }); %>
      </tr>
      """, {now: @nowTimeLabel(), columns: columns})
      tr = $(html).prependTo(tbody).hide().fadeIn('slow')

      hasAlert = _.some(columns, (c)=>
        c < @minAlertThreshold or c > @maxAlertThreshold
      )
      tr.addClass("error") if hasAlert

      timestamp = (new Date()).getTime()
      tr.data('timestamp', timestamp)

      # cleanup old rows
      tbody.find('tr').each( (index, domElem)=>
        elem = $(domElem)
        return if elem.data('timestamp') >= timestamp - @showLastNMin*60*1000
        elem.fadeOut('slow', ->
          @remove()
        )
      )

  class GrGaugeView extends GrObjectView
    initialize: ->
      super

      @gauge = null
      @decimals = 0

      properties = @model.grProperties
      inputDataProp = properties.get('inputData')
      @listenTo(inputDataProp, 'change:value', @onDataChanged) if inputDataProp

    bindPropNames: ->
      _.union(super, ['inputData'])

    renderImage: ->
      @

    render: ->
      super

      properties = @model.grProperties

      rectValues = properties.propVal('rect')?.split(',')
      [width, height] = _.last(rectValues, 2)
      width = _.str.toNumber(width)
      height = _.str.toNumber(height)

      if properties.propVal('decimals')
        @decimals = _.str.toNumber(properties.propVal('decimals'))
      minVal = _.str.toNumber(properties.propVal('minOfRange'), @decimals)
      maxVal = _.str.toNumber(properties.propVal('maxOfRange'), @decimals)

      outerCircleColor = @unpackColorData(properties.propVal('outerCircleColor'))
      outerCircleColor = "rgba(#{outerCircleColor})" unless _.str.isBlank(outerCircleColor)
      innerCircleColor = @unpackColorData(properties.propVal('innerCircleColor'))
      innerCircleColor = "rgba(#{innerCircleColor})" unless _.str.isBlank(innerCircleColor)
      opts =
        size: Math.min(width, height)
        label: properties.propVal('label')
        min: minVal
        max: maxVal
        decimals: @decimals
        zones: @applyZones(minVal, maxVal)
        outerCircleColor: outerCircleColor
        innerCircleColor: innerCircleColor
      @gauge = new Gauge(@$el[0], opts)

      initVal = _.str.toNumber(properties.propVal('initialVal'), @decimals)
      initVal = minVal if initVal < minVal or initVal > maxVal
      @gauge.write(initVal)

    applyZones: (min, max)->
      result = []
      for index in [1, 2]
        result.push(@applyZone(index, min, max))
      result

    applyZone: (index, min, max)->
      properties = @model.grProperties
      start = _.str.toNumber(properties.propVal("startOfZone#{index}"))
      end = _.str.toNumber(properties.propVal("endOfZone#{index}"))
      return null if start == end
      if start > end
        [start, end] = [end, start]

      classes = ['green', 'yellow', 'red']
      zone = {clazz : "#{classes[index]}-zone"}
      length = max - min
      zone.from = (start-min)/length
      zone.to = (end-min)/length
      return zone

    onDataChanged: (model, value, options) =>
      @gauge?.write(_.str.toNumber(value, @decimals))

  class RGraphBase extends GrObjectView
    initialize: ->
      super

      @graph = null

      properties = @model.grProperties
      @decimals = 0
      if properties.propVal('decimals')
        @decimals = _.str.toNumber(properties.propVal('decimals'))

      @canvasId = @id + '_canvas'

      inputDataProp = properties.get('inputData')
      @listenTo(inputDataProp, 'change:value', @onDataChanged) if inputDataProp

    bindPropNames: ->
      _.union(super, ['inputData'])

    renderImage: ->
      @

    render: ->
      super

      @$el.append("<canvas id='#{@canvasId}' width='#{@$el.width()}' height='#{@$el.height()}'>#{L('[No canvas support]')}</canvas>")
      setTimeout(@doRender, 0)
      @

    initVal: ->
      properties = @model.grProperties
      initVal = _.str.toNumber(properties.propVal('initialVal'), @decimals)
      initVal = @minVal if initVal < @minVal or initVal > @maxVal
      initVal

    onDataChanged: (model, value, options) =>
      @doRender(_.str.toNumber(value, @decimals))

  class RGraphThermometer extends RGraphBase
    initialize: ->
      super

      properties = @model.grProperties
      @minVal = _.str.toNumber(properties.propVal('minOfRange'), @decimals)
      @maxVal = _.str.toNumber(properties.propVal('maxOfRange'), @decimals)

      @labelColor = @unpackColorData(properties.propVal('labelColor'))
      @labelColor = "rgba(#{@labelColor})" unless _.str.isBlank(@labelColor)

      @boundingColor = @unpackColorData(properties.propVal('labelBoundingColor'))
      @boundingColor = "rgba(#{@boundingColor})" unless _.str.isBlank(@boundingColor)

      if properties.propVal('labelBounding')
        @boundingEnabled = _.str.toBoolean(properties.propVal('labelBounding'))
      else
        @boundingEnabled = false

    doRender: (value)=>
      if value
        value = _.str.toNumber(value, @decimals)
      else
        value = @initVal()

      if @graph
        RGraph.Clear(@graph.canvas)
        @graph.value = value
      else
        @graph = new RGraph.Thermometer(@canvasId, @minVal, @maxVal, value)
        @graph.Set('value.label.decimals', @decimals)
        @graph.Set('text.color', @labelColor)
        @graph.Set('value.label.bounding', @boundingEnabled)
        @graph.Set('value.label.bounding.fill', @boundingColor)
      RGraph.Effects.Thermometer.Grow(@graph)

  class RGraphProgressBar extends RGraphBase
    initialize: ->
      super

      properties = @model.grProperties
      @minVal = _.str.toNumber(properties.propVal('minOfRange'), @decimals)
      @maxVal = _.str.toNumber(properties.propVal('maxOfRange'), @decimals)

      @barColor = @unpackColorData(properties.propVal('barColor'))
      @barColor = "##{rgbToHex(@barColor)}"

      @backgroundColor = @unpackColorData(properties.propVal('backgroundColor'))
      @backgroundColor = "rgba(#{@backgroundColor})" unless _.str.isBlank(@backgroundColor)

    doRender: (value)=>
      if value
        value = _.str.toNumber(value, @decimals)
      else
        value = @initVal()

      if @graph
        RGraph.Clear(@graph.canvas)
        @graph.value = value
      else
        @graph = @createGraph(value)
        @graph.Set('background.color', @backgroundColor)
        @graph.Set('colors', ["Gradient(white:#{@barColor})"])
        @graph.Set('tickmarks', 100)
        @graph.Set('numticks', 20)
        @graph.Set('gutter.right', 30)
        @graph.Set('margin', 5)

      @drawGraph(@graph)

  class RGraphVProgressBar extends RGraphProgressBar
    createGraph: (value)->
      new RGraph.VProgress(@canvasId, @minVal, @maxVal, value)

    drawGraph: (graph)->
      RGraph.Effects.VProgress.Grow(graph)

  class RGraphHProgressBar extends RGraphProgressBar
    createGraph: (value)->
      new RGraph.HProgress(@canvasId, @minVal, @maxVal, value)

    drawGraph: (graph)->
      RGraph.Effects.HProgress.Grow(graph)

  class RGraphMeterBase extends RGraphBase
    initialize: ->
      super

      properties = @model.grProperties
      @minVal = _.str.toNumber(properties.propVal('minOfRange'), @decimals)
      @maxVal = _.str.toNumber(properties.propVal('maxOfRange'), @decimals)

    getStartEndVal: (propName)->
      properties = @model.grProperties
      strVal = _.str.trim(properties.propVal(propName))
      if _.str.endsWith(strVal, '%')
        return @minVal + (@maxVal - @minVal) * _.str.toNumber(_.str.rtrim(strVal, '%')) / 100.0
      else
        return Math.min(Math.max(_.str.toNumber(strVal), @minVal), @maxVal)

    configGraph: (graph)->
      graph.Set('red.start', @getStartEndVal('redStartVal'))
      graph.Set('red.end', @getStartEndVal('redEndVal'))
      graph.Set('yellow.start', @getStartEndVal('yellowStartVal'))
      graph.Set('yellow.end', @getStartEndVal('yellowEndVal'))
      graph.Set('green.start', @getStartEndVal('greenStartVal'))
      graph.Set('green.end', @getStartEndVal('greenEndVal'))

    clearGraph: (graph)->
        RGraph.Clear(graph.canvas)

    doRender: (value)=>
      if value
        value = _.str.toNumber(value, @decimals)
      else
        value = @initVal()

      if @graph
        @clearGraph(@graph)
        @graph.value = value
      else
        @graph = new RGraph.Meter(@canvasId, @minVal, @maxVal, value)
        @configGraph(@graph)

      RGraph.Effects.Meter.Grow(@graph)

  class RGraphMeter extends RGraphMeterBase

  class RGraphSimpleMeter extends RGraphMeterBase
    configGraph: (graph)->
      super

      graph.Set('angles.start', PI + 0.5)
      graph.Set('angles.end', TWOPI - 0.5)
      graph.Set('linewidth.segments', 5)
      graph.Set('text.size', 16)
      graph.Set('strokestyle', 'white')
      graph.Set('segment.radius.start', 175)
      graph.Set('border', 0)
      graph.Set('tickmarks.small.num', 0)
      graph.Set('tickmarks.big.num', 0)

    clearGraph: (graph)->
      RGraph.Clear(graph.canvas, 'white')

  class IFrameWidget extends GrObjectView
    initialize: ->
      super

    renderImage: ->
      @

    render: ->
      super
      properties = @model.grProperties
      src = properties.propVal('src')
      # src = 'http://' + src unless (_.str.startsWith(src, 'http') || _.str.startsWith(src, '/'))
      border = properties.propVal('border') ? 0
      @$el.append("<iframe src='#{src}' width='#{@$el.width()}' height='#{@$el.height()}' style='border: #{border};'>#{L('[No iframe support]')}</iframe>")
      @
      
  # NOTE: when add any new method, need to copy over it to
  # data/default_config/Graphics/Dashboard/dashboard_widget_template.js, so
  # that it can be used by converted widget for dashboard
  class ProxyObjectData
    constructor: (model)->
      @model = model
      @_loaded = false
      @datas = {}

    hasWritePerm: ->
      window.grCanvasView.model.actionPermitted
      
    hasData: (dataName)->
      @model.grProperties.hasProp(dataName)

    readData: (dataName)->
      @model.grProperties.propVal(dataName)
      
    invokeAction: (actionPath, value, valueDataType="void", settings={})->
      return logAndReturn(L("permission denied")) unless @hasWritePerm()

      return logAndReturn(L("invalid actionPath: ") + actionPath) unless actionPath?
      compPool = ComponentsPool.instance()
      if isUserProp(actionPath)
        valueDataType = @model.grProperties.propVal(actionPath, 'actionDataType') or valueDataType
        actionPath = @readData(actionPath)
        return logAndReturn(L("invalid actionPath: ") + actionPath) unless actionPath?
      else
        # try to get action slot's data type first, 
        # if fails, use the default 'valueDataType'(void)
        [compPath, slotName] = actionPath.split(".")
        comp = compPool.compByCompPath(compPath)

        if comp?
          typeInfo = CompTypeInfosSingleton.instance().get(comp.get('typeName'))
          if typeInfo?
            slotType = typeInfo.getAction(slotName)
            valueDataType = slotType.get('type') if slotType?
          valueDataType = 'Buf' if valueDataType == 'str'

      data =
        path: compPool.buildUrl(actionPath)
        slotType: 'action'
        type: valueDataType
        value: value

      params =
        type: "POST"
        url: compPool.proxyUrl()
        data: data
        dataType: 'json'
        success: (data)=>
          if data.error
            console.error("failed to invoke action: #{data.error.text}")
          else if data.redirect?
            redirect(data.redirect)
          else
            resp = data.response
            if resp?.resultCode != 0
              console.error("failed to invoke action #{actionPath}: " + _.str.join(",", ("path: #{error.path} desc: #{error.description}" for error in resp.errors)))
            else
              settings['success']() if settings['success']?
        
      params['beforeSend'] = settings['beforeSend'] if settings['beforeSend']
      params['complete'] = settings['complete'] if settings['complete']
      params['error'] = settings['error'] if settings['error']
        
      $.ajax params

    writeData: (dataName, dataValue, settings={})->
      return L("permission denied") unless @hasWritePerm()

      compPool = ComponentsPool.instance()
      vevent = @model.grValueEvents.get(dataName)
      dataSource = vevent.grDataSource
      comp = dataSource.component()
      typeInfo = dataSource.compTypeInfo()

      slotTypeName = "property"
      slotType = typeInfo.getProperty(dataSource.slotName())
      unless slotType
        slotType = typeInfo.getAction(dataSource.slotName())
        slotTypeName = "action"

      return L("can not find slot type data for '{dataName}'").supplant(dataName: dataName) unless slotType
       
      url = compPool.urlFor(comp, slotType.get('name'))

      # TODO: move following logics into SlotType class, add a 'getType' method
      slotDataType = slotType.get('type')
      slotDataType = 'Buf' if slotDataType == 'str'
      
      params =
        type: "POST"
        url: compPool.proxyUrl()
        data:
          path: url
          type: slotDataType
          value: dataValue
          slotType: slotTypeName
        dataType: 'json'
        success: (data)=>
          if data.error
            console.error("failed to change data slot: #{data.error.text}")
          else if data.redirect?
            redirect(data.redirect)
          else
            resp = data.response
            if resp?.resultCode != 0
              console.error("failed to write data #{url}: " + _.str.join(",", ("path: #{error.path} desc: #{error.description}" for error in resp.errors)))
            else
              settings['success']() if settings['success']?
        
      params['beforeSend'] = settings['beforeSend'] if settings['beforeSend']
      params['complete'] = settings['complete'] if settings['complete']
      params['error'] = settings['error'] if settings['error']
        
      $.ajax params
      
    runSqlQuery: (sql, dataHandler, responseFormat="json", settings={})->
      sql = _.str.trim(sql)
      unless (is_admin() || _.str.startsWith(sql.toLowerCase(), "select"))
        console.warn("only select sql statement(readonly) is permitted.")
        return

      params =
        type: "GET"
        url: "sql_query.php"
        data:
          sql: sql
          responseFormat: responseFormat
        dataType: 'json'
        success: (data)=>
          if data.error
            console.error("failed to run sql (#{sql}), error: #{data.error.text}")
          else if data.redirect?
            redirect(data.redirect)
          else
            result = data.data
            dataHandler(result) if dataHandler?

            settings['success']() if settings['success']?

      params['beforeSend'] = settings['beforeSend'] if settings['beforeSend']
      params['complete'] = settings['complete'] if settings['complete']
      params['error'] = settings['error'] if settings['error']

      $.ajax params
      
    readNote: (path, dataHandler, settings={}) ->
      params =
        type: "GET"
        url: "note_controller.php"
        data:
          path: path
        dataType: 'json'
        success: (data)=>
          if data.error
            settings['error']() if settings['error']?
            console.error("failed to get note (#{path}), error: #{data.error.text}")
          else if data.redirect?
            redirect(data.redirect)
          else
            content = data.content
            dataHandler(content) if dataHandler?

            settings['success']() if settings['success']?

      params['beforeSend'] = settings['beforeSend'] if settings['beforeSend']
      params['complete'] = settings['complete'] if settings['complete']
      params['error'] = settings['error'] if settings['error']

      $.ajax params
      
    writeNote: (path, content, settings={}) ->
      return "permission denied" unless @hasWritePerm()

      params =
        type: "POST"
        url: "note_controller.php"
        data:
          path: path
          content: content
        dataType: 'json'
        success: (data)=>
          if data.error
            settings['error']() if settings['error']?
            console.error("failed to save note (#{path}): #{data.error.text}")
          else if data.redirect?
            redirect(data.redirect)
          else
            resp = data.response
            settings['success']() if settings['success']?
        
      params['beforeSend'] = settings['beforeSend'] if settings['beforeSend']
      params['complete'] = settings['complete'] if settings['complete']
      params['error'] = settings['error'] if settings['error']
        
      $.ajax params
      
    L: (str)->
      L(str)
      
    getExportedClass: (clsName)->
      return GrDateTimePicker if clsName == 'GrDateTimePicker'
      
    startSpinner: ->
      @spinner = new Spinner() unless @spinner?
      @spinner.spin(@elem[0]) if @elem? and @elem.length > 0
      
    stopSpinner: ->
      @spinner?.stop()
      
    handleColorData: (cr)->
      return "" unless cr
      return cr if cr.startsWith('#')

      parts = cr.split(",")
      if parts.length == 4 #there is alpha channel data 
        parts[3] = (parts[3]/255)
        "rgba(#{parts.join(",")})"
      else if parts.length == 3
        "rgb(#{parts.join(",")})"
      else
        ""
        
    parseEnumData: (dataName)->
      choices = @readData(dataName)?.split(',')
      return [] unless choices

      choice_default_val = 0
      _.map(choices, (c)->
        obj = _.object(['label', 'value'], c.split(':'))
        obj.value = parseFloat(obj.value)
        if isNaN(obj.value)
          obj.value = choice_default_val
        else
          choice_default_val = obj.value
        ++choice_default_val
        obj
      )

    slotDataType: (dataName)->
      vevent = @model.grValueEvents.get(dataName)
      dataSource = vevent.grDataSource
      typeInfo = dataSource.compTypeInfo()

      typeData = typeInfo.getProperty(dataSource.slotName())
      unless typeData
        typeData = typeInfo.getAction(dataSource.slotName())
       
      slotDataType = typeData.get('type')
      slotDataType = 'Buf' if slotDataType == 'str'
      slotDataType
      
    isAction: (dataName)->
      vevent = @model.grValueEvents.get(dataName)
      dataSource = vevent.grDataSource
      typeInfo = dataSource.compTypeInfo()

      typeData = typeInfo.getProperty(dataSource.slotName())
      return !typeData
    
    isDashboard: ->
      return false

  class AdapterWidget extends GrObjectView
    initialize: ->
      super

      @proxyObj = new ProxyObjectData(@model)

      properties = @model.grProperties
      for index in [1..16]
        @installUserPropListeners("data#{index}")
        
      @installUserPropListeners(userProp) for userProp in properties.userProps()

      jsCode = properties.get("javascriptCode").get('value')
      @widgetName = @model.get('name')

      (new Function(jsCode)).call(@proxyObj)

      @requiredScripts = []
      if @proxyObj.hasOwnProperty("requiredScripts")
        @requiredScripts = _.chain([@proxyObj['requiredScripts']()]).flatten().compact().value()
        
    installUserPropListeners: (propName)->
      properties = @model.grProperties
      @proxyObj.datas[propName] = properties.propVal(propName)
      dataProp = properties.get(propName)
      @listenTo(dataProp, 'change:value', @onDataChanged) if dataProp

    bindPropNames: ->
      _.union(super, _.keys(@proxyObj.datas))

    renderImage: ->
      @

    render: ->
      super

      @$el.css("overflow", "visible")

      properties = @model.grProperties
      @proxyObj['width'] = @$el.width()
      @proxyObj['height'] = @$el.height()
      @proxyObj['containerId'] = @widgetName + '_container'
      @proxyObj['elem'] = $("<div id='#{@proxyObj['containerId']}' style='width:#{@$el.width()}px; height:#{@$el.height()}px; margin: 0 auto'></div>").appendTo(@$el)
      
      @loadScripts()
      @

    init: ->
      try
        @proxyObj['init']() if @proxyObj.hasOwnProperty("init")
      catch err
        console.log(err.stack)
        # console.trace()
        console.warn "failed to execute 'init' method: #{err.message}"

      @proxyObj._loaded = true
      
    loadScripts: =>
      if @requiredScripts.length == 0
        @init()
      else
        scriptUrl = @requiredScripts.shift()
        ScriptLoaderSingleton.instance().load(scriptUrl, @loadScripts, @loadFailed)
        
    loadFailed: (scriptUrl)=>
      console.warn "failed to load required script: " + scriptUrl

    onDataChanged: (model, value, options)=>
      name = model.get('name')
      @proxyObj.datas[name] = value

      return unless @proxyObj._loaded

      try
        @proxyObj['update'](name) if @proxyObj.hasOwnProperty("update")
      catch err
        console.warn "failed to execute 'update' method: #{err.message}"

  class ActionButtonView extends GrObjectView

    initialize: ->
      super

    renderImage: ->
      @

    render: ->
      super

      properties = @model.grProperties
      label = properties.propVal('label')
      image = properties.propVal('image')
      width = @$el.width()
      height = @$el.height()
      
      if _.str.endsWith(image, "web_action_button.png")
        @$el.append("<button type='button' class='btn' style='width:#{width}px; height:#{height}px;'><span>#{label}</span></button>")
      else
        @$el.append("<div style='width:#{width}px; height:#{height}px; background-image: url(#{image}); background-repeat:no-repeat; background-size:#{width-2}px #{height-2}px; cursor:pointer; display:table;'><span style='display:table-cell; text-align:center; vertical-align:middle;'>#{label}</span></div>")

      font = properties.propVal('font')
      color = properties.propVal('textColor')
      txtElem = @$el.find("span")
      TextFormatter.applyFontInfo txtElem, font
      TextFormatter.applyTextColorInfo txtElem, color

      @

    initMenu: ->
      return unless @hasWritePerm()
      @$el.on('mousedown', @onMouseDown)
      @$el.on('mouseup', @onMouseUp)
      @$el.on('click', @invokeAction)

    invokeAction: =>
      properties = @model.grProperties
      src = properties.propVal('clickAction')

      ds = @model.grValueEvents.get('clickAction')?.grDataSource
      return unless ds

      typeInfo = ds.compTypeInfo()
      return unless typeInfo

      slot = typeInfo.getAction(ds.get('slotName'))
      return unless slot

      compPool = ComponentsPool.instance()
      comp = compPool.compByCompPath(ds.get("compPath"))
      return unless comp

      menuItemData =
        type: 'action'
        label: L('Invoke Action')
        slotType: slot
        comp: comp

      @askParameterDialog.menuItemData = menuItemData
      @askParameterDialog.render()

    onMouseDown: =>
      return if @$el.find('button').length != 0

      @$el.css("margin-top", "+=2")
      @$el.css("margin-left", "+=2")

    onMouseUp: =>
      return if @$el.find('button').length != 0

      @$el.css("margin-top", "-=2")
      @$el.css("margin-left", "-=2")
      
  class DateTimeBaseView extends GrLabelView

    initialize: ->
      super
      @formatIndex = _.str.toNumber(@model.grProperties.propVal('format'))
      @listenTo(DateTimeUtilitySingleton.instance(), "dateTimeUpdated", @updateLabel)
      @listenToOnce(grCanvasView, "canvasRendered", DateTimeUtilitySingleton.instance().startPolling)

    renderImage: ->
      @

    isPureTextLabel: ->
      true
      
    updateLabel: =>
      @$('.grText').text(@getLabelText())
      
    getLabelText: ->
      instance = DateTimeUtilitySingleton.instance()
      d = moment(
        year: parseInt(instance.get('year')?.value)
        month: parseInt(instance.get('month')?.value)-1
        day: parseInt(instance.get('day')?.value)
        hour: parseInt(instance.get('hour')?.value)
        minute: parseInt(instance.get('minute')?.value)
        second: parseInt(instance.get('second')?.value)
      )
      console.debug("delay is: " + instance.networkDelay)
      d.add(instance.networkDelay, "ms")
      @renderDateTimeString(d)

  class DateView extends DateTimeBaseView

    renderDateTimeString: (d)->
      return L("Loading...") unless d.isValid()
      switch @formatIndex
        when 0 # DD/MM/YYYY
          d.format("DD/MM/YYYY")
        when 1 # MM/DD/YYYY
          d.format("MM/DD/YYYY")
        when 2 # M D,YYYY
          d.format("MMM D,YYYY")
        when 3 # D M,YYYY
          d.format("D MMM,YYYY")
        else
          d.format("DD/MM/YYYY")

  class TimeView extends DateTimeBaseView

    renderDateTimeString: (d)->
      return L("Loading...") unless d.isValid()
      switch @formatIndex
        when 0 # HH:MM
          d.format("HH:mm")
        when 1 # HH:MM AM/PM
          d.format("HH:mm A")
        else
          d.format("HH:mm")

  class GrCanvasView extends Backbone.View
    el: 'div#grCanvas'

    initialize: ->
      _.bindAll @
      @model = new GrDoc
      @objViews = []
      @interval = getUrlParam("interval") ? 3000
      @batchReqInterval = getUrlParam("batch_interval") ? 10
      @rendered = false
      @retryTimes = 0

      @listenTo(GrNavInstance.instance(), "change", @loadDocInfo)
      @listenTo(@model, "docInfoLoaded", @loadDocData)
      @listenTo(@model, "sync", @onDocDataLoaded)

      @listenTo(ComponentsPool.instance(), "remove", @onComponentRemoved)

      @spinner = new Spinner({top: 200})
      @spinner.spin(document.body)

    loadDocInfo: =>
      window.cpt_timing_data['grDocDataLoadStart'] = (new Date()).getTime()
      @model.fetchDocInfo()

    loadDocData: =>
      @model.fetch({
        success: (model, response, options)->
          console.info("doc data is loaded successfully")
        error: (model, response, options)->
          alert(L("failed to fetch graphics data"))
      }).done(->
        # console.info "doc data loading is done"
      ).fail((jqXHR, textStatus, errorThrown)->
        console.error errorThrown
      ).always(->
        window.cpt_timing_data['grDocDataLoadEnd'] = (new Date()).getTime()
      )

    onDocDataLoaded: ->
      @listenToOnce(ComponentsPool.instance(), "dataload", =>
        _.defer(@render)
      )
      window.cpt_timing_data['bindingDataLoadStart'] = (new Date()).getTime()
      @loadData()

    render: ->
      return if @rendered
      @rendered = true

      window.cpt_timing_data['bindingDataLoadEnd'] = (new Date()).getTime()
      @spinner.stop()

      return unless (canvas = @model.get('grCanvas'))?

      @$el.empty()

      # increase container's width based on canvas
      width = _.str.toNumber(canvas.get('width'))
      $('.container').width($('#leftSideBar').width() + width + 50)

      @$el.css({
        "left": "#{canvas.get('x') ? 0}px",
        "top": "#{canvas.get('y') ? 0}px",
        "height": "#{canvas.get('height')}px",
        "width": "#{width}px",
        "background-color": "rgba(#{canvas.get('backgroundColor') ? '229, 227, 223, 255'})"
      })

      $("<div id='grContainer'></div>").appendTo(@$el)

      @renderLayer(layer) for layer in canvas.grLayers?.models
      
      console.info("canvas rendered")
      @trigger("canvasRendered")
      
      window.cpt_timing_data['canvasRendered'] = (new Date()).getTime()
      window.cpt_timing_data.logLoadingTime()

      @

    renderLayer: (layer) ->
      @renderObject(object) for object in layer.grObjects?.models.reverse()
      
    createObjectView: (object)->
      objectView = null
      return objectView unless object?

      widgetClass = object.get("widgetClass")
      dataType = object.get("dataType")
      options = {model: object, id: object.get("name")}
      switch widgetClass
        when "GrWebLineEdit"
          validationIndex = _.str.toNumber(object.grProperties.propVal('validation'))
          # validationRange = object.grProperties.propVal('validation', 'range')
          # validationRange = _.map(_.str.words(validationRange, ','), (w)->
          #   _.str.trim(w)
          # )
          switch validationIndex
            when 1 # IP Address validation
              objectView = new GrWebIPAddressEditView(options)
            when 2 # Time Range validation
              objectView = new GrWebTimeRangeEditView(options)
            when 3 # Date Range validation
              objectView = new GrWebDateRangeEditView(options)
            when 4 # Number Range validation
              objectView = new GrWebNumberRangeEditView(options)
            when 5 # Hex Number validation
              objectView = new GrWebHexNumberEditView(options)
            when 6 # Time Validation
              objectView = new GrWebTimeEditView(options)
            else
              if dataType == "WebPasswordEdit"
                objectView = new GrWebPasswordEditView(options)
              else
                objectView = new GrWebLineEditView(options)
        when "GrDateRangeWebLineEdit"
          objectView = new GrWebDateRangeEditView(options)
        when "GrWebSliderEdit"
          objectView = new GrWebSliderEditView(options)
        when "GrWebDateTimeEdit"
          objectView = new GrWebDateTimeEditView(options)
        when "GrLabel"
          objectView = new GrLabelView(options)
        when "GrWidget"
          objectView = new GrObjectView(options)
        when "GrDynamicWebWidgetEdit"
          switch dataType
            when 'LineChart'
              objectView = new GrLineChartView(options)
            when 'HistoryLineChart'
              objectView = new GrHistoryLineChartView(options)
            when 'Table'
              objectView = new GrTableView(options)
            when 'Gauge'
              objectView = new GrGaugeView(options)
            when 'Thermometer'
              objectView = new RGraphThermometer(options)
            when 'VProgressBar'
              objectView = new RGraphVProgressBar(options)
            when 'HProgressBar'
              objectView = new RGraphHProgressBar(options)
            when 'Meter'
              objectView = new RGraphMeter(options)
            when 'SimpleMeter'
              objectView = new RGraphSimpleMeter(options)
            when 'iFrameWidget'
              objectView = new IFrameWidget(options)
            when 'AdapterWidget'
              objectView = new AdapterWidget(options)
            when 'ActionButton'
              objectView = new ActionButtonView(options)
            when 'DateLabel'
              objectView = new DateView(options)
            when 'TimeLabel'
              objectView = new TimeView(options)
            else
              objectView = new GrObjectView(options)
        else
          objectView = new GrObjectView(options)

      return objectView

    renderObject: (object) ->
      return if ((not object.isVisible()) and (not object.dataSource()?))
      return unless object.isDataValid()

      objectView = @createObjectView(object)
      return unless objectView?

      objectView.render()
      @$("#grContainer").append(objectView.el)
      @objViews.push(objectView)

    deLayedLoadData: =>
      instance = ComponentsPool.instance()
      if instance.isBatchReqDone()
        # check if there are valid urls, if no, just stop the data loading
        # process
        return if instance.isCompUrlEmpty()
        _.delay(@loadData, @interval, {remove: false})
      else
        _.delay(@loadData, @batchReqInterval, {remove: false})

    onFetchDataError: (jqXHR, status, error)=>
      if not @rendered # before canvas rendered
        if @retryTimes < 5
          @retryTimes += 1
          console.warn("failed to fetch & parse component data, retry " + @retryTimes)
          @deLayedLoadData()
        else
          # restore @retryTimes
          @retryTimes = 0
          alert(L("failed to fetch & parse component data from controller. please make sure the latest version of easyioCpt kit installed."))
          @spinner.stop()
      else # after canvas rendered
        console.warn("failed to fetch & parse component data, retry...")
        @deLayedLoadData()

    loadData: =>
      instance = ComponentsPool.instance()
      if instance.length == 0
        #FIXME: render maybe called twice when there is no component binding on
        #page, 'dataLoad' signal trigger first, and here will be called the
        #second time
        @render()
      else
        instance.fetch
          remove: false
          success: @deLayedLoadData
          error: @onFetchDataError

    onComponentRemoved: (component, collection, options)->
      removedViews = []
      for view in @objViews
        dataSource = view.model.dataSource()
        continue unless dataSource?
        continue unless compPath2compId(dataSource.get('compPath')) == component.id
        switch dataSource.get('failOver')
          when 'fade'
            view.$el.css({'opacity': 0.7})
          else
            view.$el.fadeOut(400, view.$el.remove)
        removedViews.push(view)
      @objViews = _.difference(@objViews, removedViews)

  class GrNavControlView extends Backbone.View
    el: 'div#navSideBarControlBtn'
    
    events:
      'click' : 'toggleNavSideBar'
      'mouseenter': 'highlight'
      'mouseleave': 'unhighlight'
    
    render: ->
      @$el
      
    toggleNavSideBar: ->
      $("#leftSideBar").toggle('fast', =>
        @$el.find("a").text($("#leftSideBar").is(":visible") and '-' or '+')
      )
      
    hideNavSideBar: ->
      $("#leftSideBar").hide()
      @$el.find("a").text('+')
      
    showNavSideBar: ->
      $("#leftSideBar").show()
      @$el.find("a").text('-')
      
    highlight: ->
      @$el.removeClass("unhighlight").addClass("highlight")
        
    unhighlight: ->
      @$el.removeClass("highlight").addClass("unhighlight")


  class GrNavView extends Backbone.View
    el: 'div#grNav'

    initialize: ->
      _.bindAll @

      @model = GrNavInstance.instance()
      @listenTo(@model, "change", @render)

      @navControlView = new GrNavControlView

      window.cpt_timing_data['grNavDataLoadStart'] = (new Date()).getTime()
      @model.fetch({
        dataType: 'json',
        success: (model, response, options)->
          console.info "nav data loaded"
          window.cpt_timing_data['grNavDataLoadEnd'] = (new Date()).getTime()
          model.trigger("change")
        error: (model, response, options)->
          console.error("failed to load navigation menu data")
          window.cpt_timing_data['grNavDataLoadEnd'] = (new Date()).getTime()
          alert("failed to load navigation menu data")
      })

    render: ->
      @$el.empty()
      
      if @model.showAtStart()
        @navControlView.showNavSideBar()
      else
        @navControlView.hideNavSideBar()

      ul = $("<ul class='nav nav-list'></ul>").appendTo(@$el)

      @renderGroup(@model.topGroup, ul)
      
      @$el
        .on 'changed.jstree', (e, data)->
          return unless data.node?

          if data.instance.is_leaf(data.node)
            containerElem = $(".container")[0]
            new Spinner({top: "200px"}).spin(containerElem) if containerElem
            redirect(data.node.a_attr.href)
          else
            data.instance.toggle_node(data.node)
        .on 'load_node.jstree', (e, data)=>
          selectedRow = @$el.find('.jstree-wholerow-clicked')
          selectedRow.height(selectedRow.parent().children('a.jstree-anchor').height())
        .jstree
          "core":
            "multiple": false
            "themes":
              "dots": false
              "responsive": false
          "plugins": ["wholerow"]
           
      @$el.on 'hover_node.jstree', =>
        hoveredRow = @$el.find('.jstree-wholerow-hovered')
        hoveredRow.height(hoveredRow.parent().children('a.jstree-anchor').height())

      window.cpt_timing_data['grNavRendered'] = (new Date()).getTime()

      @$el

    isGrOnBlackList: (item)->
      blackList = @model.grBlackList
      return _.contains(blackList, item.path())

    icon: (item)->
      if item.icon() == "icon-home"
        return "../js/jstree/themes/default/home.png"
      else if item.icon() == "icon-picture"
        return "jstree-file"
      else
        return "jstree-folder"

    renderGroup: (group, parent)->
      return if @isGrOnBlackList(group)

      groupLi = $("""
      <li data-jstree='{"icon": "#{@icon(group)}"}'>
          <a href='#'> #{ group.get('name') }</a>
      </li>""").appendTo(parent)

      ul = $("<ul class='nav nav-list'></ul>").appendTo(groupLi)

      for item in group.items
        if item.isGroup()
          @renderGroup(item, ul)
        else
          @renderItem(item, ul)

    renderItem: (item, parent)->
      return if @isGrOnBlackList(item)
      selected = item.get('path') == @model.grName()

      li = $("""
      <li data-jstree='{"selected": #{selected}, "icon": "#{@icon(item)}"}'>
          <a href='?grname=#{ encodeURIComponent(item.get('path')) }'> #{ item.get('name') }</a>
      </li>""")

      li.appendTo(parent)

  ##########################################################
  # Dialog Views
  ##########################################################

  class AlertMessageView extends Backbone.View
    el: 'div#alertMessage'

    render: (title, message)->
      html = $("#modal_tpl").html()
      @$el.html(_.template(html, {title: title, message: "<p>#{message}</p>", btnHtml: ''}))
      @$el.modal()
      @

  class UtilityRestoreView extends Backbone.View
    tagName: 'div'
    className: "modal hide fade"
    attributes:
      'data-backdrop': 'static'

    initialize: ->
      @spinner = null

    render: ->
      @$el.empty()
      html = $("#modal_tpl").html()
      @$el.html(_.template(html, {
        title: L('Restore <small>sedona app & graphics</small>')
        message: ''
        btnHtml: ''
      }))
      @$el.modal()

      data =
        'action': 'list_backups'

      $.ajax
        type: 'GET'
        url: "utility.php"
        dataType: 'json'
        data: data
        success: @onDataLoaded
        error: (data, status)=>
          @$el.modal('hide')
          view = new AlertMessageView
          view.render(L("Error"), status)

    doRenderBackups: (body, backups, type)->
      backups = _.filter backups, (b)->
        b.type == type
      
      return if backups.length == 0

      if (type == "flash")
        $("<h4>"+L("Backups in Flash")+"</h4>").appendTo(body)
      else if (type == "sdcard")
        $("<h4>"+L("Backups in SDCard")+"</h4>").appendTo(body)

      table = $("""<table class="table table-striped table-hover"></table>""").appendTo(body)
      for b in backups
        d = new Date(b.ts)
        dateLabel = "#{d.getFullYear()}/#{d.getMonth()+1}/#{d.getDate()}"
        $("""
        <tr>
            <td class="backup-name" data-type="#{type}">#{b.name}</td>
            <td style='display: none;'>#{dateLabel}</td>
            <td>
                <button class="btn btn-danger pull-right delete-btn" style="margin-left: 10px;">#{L('Delete')}</button>
                <button class="btn pull-right restore-btn">#{L('Restore')}</button>
            </td>
        </tr>""").prependTo(table)
      table.find('.restore-btn').click(@doRestore)
      table.find('.delete-btn').click(@doDelete)

    renderBackups: (backups)->
      body = @$el.find('.modal-body')
      if _.isEmpty(backups)
        $(L("<p>There is no backup yet.</p>")).appendTo(body)
      else
        @doRenderBackups(body, backups, "flash")
        @doRenderBackups(body, backups, "sdcard")

    onServerReady: =>
      @$el.modal('hide')
      location.reload()

    onServerTimeout: =>
      @$el.find('.alert').removeClass('alert-success').addClass('alert-error').html(L("<h4>Can not connect to server!</h4>")).show()

    doRestore: (e)=>
      btnElem = $(e.target)
      backupElem = btnElem.parent("td").siblings('td.backup-name')
      name = backupElem.text()
      type = backupElem.data("type")

      return if _.str.isBlank(name)

      confirmed = confirm(L("Are you sure to restore to '{name}' ?").supplant(name: name))
      return unless confirmed

      url = "utility.php"
      $.ajax
        type: "POST"
        url: url
        data:
          'action': 'restore'
          'data[name]': name
          'data[type]': type
        dataType: 'json'
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
          @$el.find('.restore-btn').addClass("disabled").attr("disabled", "disabled")
          @$el.find('.delete-btn').addClass("disabled").attr("disabled", "disabled")
          @$el.find('.alert').removeClass('alert-error').removeClass('alert-succes').text('').hide()
        success: (data)=>
          if data.error
            @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').addClass('alert-success').html(L("<h4>Succeed restoring to backup'{name}'!</h4>reloading now, please wait ...").supplant(name: name)).show()

            $(window).on('serverReady', @onServerReady)
            $(window).on('serverTimeout', @onServerTimeout)
            checker = ServerStateCheckerSingleton.instance()
            checker.check(3000, 10000)

        error: (data, status)=>
          console.log "error status: " + status
          @$el.modal('hide')

          view = new AlertMessageView
          view.render(L("Error"), status)
          @spinner.stop()
        complete: (data, status)=>
          @$el.find('.restore-btn').removeClass("disabled").removeAttr("disabled")
          @$el.find('.delete-btn').removeClass("disabled").removeAttr("disabled")

    doDelete: (e)=>
      btnElem = $(e.target)
      backupElem = btnElem.parent("td").siblings('td.backup-name')
      name = backupElem.text()
      type = backupElem.data("type")
      confirmed = confirm(L("Are you sure to delete backup '{name}' ?").supplant(name: name))
      return unless confirmed

      url = "utility.php"
      $.ajax
        type: "POST"
        url: url
        data:
          'action': 'delete'
          'data[name]': name
          'data[type]': type
        dataType: 'json'
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
          @$el.find('.restore-btn').addClass("disabled").attr("disabled", "disabled")
          @$el.find('.delete-btn').addClass("disabled").attr("disabled", "disabled")
          @$el.find('.alert').removeClass('alert-error').removeClass('alert-succes').text('').hide()
        success: (data)=>
          if data.error
            @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').addClass('alert-success').html(L("<h4>Backup'{name}' is deleted!</h4>").supplant(name: name)).show()
            btnElem.parent("td").parent("tr").fadeOut(700, ->
              $(this).remove()
            )
        error: (data, status)=>
          console.log "error status: " + status
          @$el.modal('hide')

          view = new AlertMessageView
          view.render(L("Error"), status)
        complete: (data, status)=>
          @$el.find('.restore-btn').removeClass("disabled").removeAttr("disabled")
          @$el.find('.delete-btn').removeClass("disabled").removeAttr("disabled")
          @spinner.stop()

    onDataLoaded: (data)=>
      if data.redirect?
        redirect(data.redirect)
      else if data.backups
        @renderBackups(data.backups)

  class UtilityBackupView extends Backbone.View
    tagName: 'div'
    className: "modal hide fade"
    attributes:
      'data-backdrop': 'static'

    initialize: ->
      @spinner = null

    render: ->
      @$el.empty()

      d = new Date()
      month = _.str.pad(d.getMonth()+1, 2, '0')
      day = _.str.pad(d.getDate(), 2, '0')
      defaultBackupName = "#{d.getFullYear()}#{month}#{day}_backup"

      form_html = """
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label" for="inputBackupName">#{L('Folder Name:')}</label>
          <div class="controls">
            <input type="text" id="inputBackupName" value='#{defaultBackupName}' autofocus>
      """
      pname = window.platformName
      if pname != "FG" and pname != "FG+" and pname != "FG-"
        form_html += """
            <label style="margin-top: 6px;">
              <select id="inputBackupType">
        """
        if window.sdcardExisted
          form_html += """
                  <option value='sdcard' selected='selected'>#{L('Backup to SDCard')}</option>
          """
        form_html += """
                <option value='flash'>#{L('Backup to Flash')}</option>
              </select>
            </label>
            """
      form_html += """
            <label class="checkbox" style="margin-top: 6px;">
              <input type="checkbox" id="inputBackupHistoryDB" > #{L('Backup History DB')}
            </label>
          </div>
        </div>
      </form>
      """

      html = $("#modal_tpl").html()
      @$el.html(_.template(html, {
        title: L('Backup <small>sedona app & graphics</small>'),
        message: form_html,
        btnHtml: "<a href='#' class='btn btn-primary'>#{L('Backup')}</a>"
      }))
      @$el.modal()

      @$el.find('form').submit(@submitHandler)
      @$el.find('.btn-primary').on('click', @doBackup)

    validateNotEmpty: ->
      nameInput = @$el.find("#inputBackupName")

      return true unless _.str.isBlank(nameInput.val())

      renderError(nameInput, L("backup folder name can not be empty"))
      return false

    validateValidChars: ->
      nameInput = @$el.find("#inputBackupName")
      val = nameInput.val()

      error = validateNormalText(val, L("backup folder name"), 2)
      return true unless error

      renderError(nameInput, error)
      return false

    removeErrors: ->
      @$el.find('.help-inline').remove()
      @$el.find('.control-group').removeClass('error')

    doBackup: =>
      @removeErrors()
      return unless (@validateNotEmpty() and @validateValidChars())

      @spinner = new Spinner() unless @spinner
      @spinner.spin(@$el[0])
      app = AppUtiltiySingleton.instance()
      app.save(@sendRequest, =>
        @spinner?.stop()
      )

    submitHandler: (e)=>
      e.preventDefault()
      @doBackup()

    sendRequest: =>
      name = @$el.find('#inputBackupName').val()
      type = @$el.find('#inputBackupType').val()
      backupHistoryDB = @$el.find('#inputBackupHistoryDB').is(':checked')
      data =
        'action': 'backup'
        'data[name]': name
        'data[type]': type
        'data[backupHistoryDB]': backupHistoryDB

      url = "utility.php"
      $.ajax
        type: "POST"
        url: url
        data: data
        dataType: 'json'
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
          primaryBtn = @$el.find('.btn-primary')
          primaryBtn.addClass("disabled").attr("disabled", "disabled")
          btnText = primaryBtn.html()
          unless _.str.endsWith(btnText, '...')
            primaryBtn.html("#{btnText}...")
          @$el.find('.alert').removeClass('alert-success').removeClass('alert-error').text('').hide()
        success: (data)=>
          if data.error
            @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').addClass('alert-success').html(L("<h4>Backup is done.</h4>")).show()
        error: (data, status)=>
          console.log "error status: " + status
          @$el.modal('hide')

          view = new AlertMessageView
          view.render(L("Error"), status)
        complete: (data, status)=>
          primaryBtn = @$el.find('.btn-primary')
          primaryBtn.removeClass("disabled").removeAttr("disabled")

          btnText = primaryBtn.html()
          if _.str.endsWith(btnText, '...')
            primaryBtn.html(_.str.rtrim(btnText, '.'))
          @spinner?.stop()

  class UtilityBaseDialog extends Backbone.View
    tagName: 'div'
    className: "modal hide fade"
    attributes:
      'data-backdrop': 'static'

    initialize: ->
      @spinner = null

    renderConfirmationModal: (data)->
      html = $("#modal_tpl").html()
      @$el.empty()
      data = _.defaults(data, {title: '', message: '', btnHtml: ''})
      @$el.html(_.template(html, data))
      @$el.modal()

    sendRequest: (data) ->
      compPool = ComponentsPool.instance()
      data = _.defaults(data, {slotType: 'action', type: 'void', value: ''})

      $.ajax
        type: "POST"
        url: compPool.proxyUrl()
        data: data
        dataType: 'json'
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled").attr("disabled", "disabled")
          @$el.find('.alert').removeClass('alert-error').removeClass('alert-succes').text('').hide()
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
        success: @onAjaxSuccess
        error: (data, status)=>
          console.log "error status: " + status
          @$el.modal('hide')
          view = new AlertMessageView
          view.render(L("Error"), status)
        complete: (data, status)=>
          @$el.find('.btn-primary').removeClass("disabled").removeAttr("disabled")
          @spinner.stop()

  class UtilityRestartView extends UtilityBaseDialog
    render: ->
      @renderConfirmationModal({
        title: L('Restart'),
        message: L("<p>Are you sure to restart ?</p>"),
        btnHtml: "<a href='#' class='btn btn-primary'>#{L('Restart')}</a>"
      })
      @$el.find(".btn-primary").on('click',  @doRestart)

    doRestart: =>
      compPool = ComponentsPool.instance()
      path = compPool.buildUrl('/', 'restart')
      @sendRequest({path: path})

    onAjaxSuccess: (data)=>
      if data.error
        @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
      else if data.redirect?
        redirect(data.redirect)
      else
        @$el.find('.alert').addClass('alert-success').html(L("<h4>Restarted.</h4>reloading now, please wait ...")).show()
        setTimeout(
          =>
            @$el.modal('hide')
            location.reload()
          ,3000)

  class UtilityRebootView extends UtilityBaseDialog
    render: ->
      @renderConfirmationModal({
        title: L('Reboot'),
        message: L("<p>Are you sure to reboot ?</p>"),
        btnHtml: "<a href='#' class='btn btn-primary'>#{L('Reboot')}</a>"
      })
      @$el.find(".btn-primary").on('click', @doReboot)

    doReboot: =>
      compPool = ComponentsPool.instance()
      path = compPool.buildUrl('/', 'reboot')
      @sendRequest({path: path})

    onAjaxSuccess: (data)=>
      if data.error
        @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
      else if data.redirect?
        redirect(data.redirect)
      else
        @$el.find('.alert').addClass('alert-success').html(L("<h4>Rebooted.</h4>reloading now, please wait ...")).show()
        setTimeout(
          =>
            @$el.modal('hide')
            location.reload()
          ,2000)

  class UtilityUpgradeView extends Backbone.View
    tagName: 'div'
    className: "modal hide fade"
    attributes:
      'data-backdrop': 'static'

    initialize: ->
      @spinner = null

    render: ->
      @$el.empty()

      form_html = """
      <form class="form-horizontal" action="upgrade_firmware.php" method="post" enctype="multipart/form-data">
        <div class="control-group">
          <label class="control-label" for="inputFirmwareFile">#{L('Firmware File:')}</label>
          <div class="controls">
            <input type="file" id="inputFirmwareFile" name="firmwareFile">
          </div>

          <div id="firmware-upgrade-progress" class="progress progress-info progress-striped">
            <div class="bar" style="width: 0%"></div>
          </div>
        </div>
      </form>
      """

      html = $("#modal_tpl").html()
      @$el.html(_.template(html, {
        title: L('Upgrade Firmware'),
        message: form_html,
        btnHtml: "<a href='#' class='btn btn-danger disabled' disabled='disabled'>#{L('Upgrade')}</a>"
      }))
      @$el.modal()
      
      @$el.find("input:file").change @onFileSelected
      @$el.find('form').ajaxForm
        beforeSubmit: =>
          @removeErrors()
          if @validateFileInput()
            @$el.find('#firmware-upgrade-progress').show().find('.bar').width("0%")
            @$el.find('.alert').addClass('alert-success').removeClass('alert-error').html(L('<h4>Start to upload firmware file ...</h4>')).show()
            return true
          else
            return false
        uploadProgress: (event, pos, total, percentComplete)=>
          @$el.find('#firmware-upgrade-progress .bar').width("#{percentComplete}%")
        success: (responseText, statusText, xhr, form)=>
          @$el.find('#firmware-upgrade-progress .bar').width("100%")
          uploadSuccess = _.str.startsWith(responseText, "SUCCESS:")
          if uploadSuccess
            @$el.find('.alert').removeClass('alert-error').addClass('alert-success').html("<h4>#{responseText}</h4>").show()
            _.delay(@doReboot, 750)
          else
            @$el.find('.alert').removeClass('alert-success').addClass('alert-error').html("<h4>#{responseText}</h4>").show()
            @resetUpgradeState()
        complete: =>
          @$el.find('#firmware-upgrade-progress').hide()

      @$el.find('.btn-danger').on('click', @doUpgrade)
      
    onFileSelected: =>
      @removeErrors()
      @$el.find('#firmware-upgrade-progress').show().find('.bar').width("0%")

      if @validateFileInput()
        @$el.find('.btn-danger').removeClass('disabled').removeAttr('disabled')
      else
        @$el.find('.btn-danger').addClass("disabled").attr("disabled", "disabled")
      
    userSelectedFile: ->
      @$el.find('#inputFirmwareFile')?.val()

    validateFileInput: ->
      filePath = @userSelectedFile()
      if _.str.isBlank(filePath)
        @$el.find('.alert').removeClass('alert-success').addClass('alert-error').html(L("<h4>Firmware file is not specified.</h4>")).show()
        return false

      unless _.str.endsWith(filePath, window.firmwareFileName)
        @$el.find('.alert').removeClass('alert-success').addClass('alert-error').html(L("<h4>Firmware file name must be '#{ window.firmwareFileName }'.</h4>")).show()
        return false

      return true
    
    removeErrors: ->
      @$el.find('.alert').removeClass('alert-error').hide()
      @$el.find('.help-inline').remove()
      @$el.find('.control-group').removeClass('error')
      
    resetUpgradeState: ->
      primaryBtn = @$el.find('.btn-danger')
      primaryBtn.removeClass("disabled").removeAttr("disabled")
      
      btnText = primaryBtn.html()
      if _.str.endsWith(btnText, '...')
        primaryBtn.html(_.str.rtrim(btnText, '.'))
      @spinner?.stop()
      
    onServerReady: =>
      @$el.find('.alert').removeClass('alert-error').addClass('alert-success').html(L("<h4>Controller is rebooted, reload page now ...</h4>")).show()
      _.delay(=>
        @resetUpgradeState()
        location.reload()
      , 750)
      
    onServerTimeout: =>
      @$el.find('.alert').removeClass('alert-success').addClass('alert-error').html(L("<h4>Controller timeout.</h4>")).show()
      @resetUpgradeState()

    onRebootSuccess: =>
      @$el.find('.alert').removeClass('alert-error').addClass('alert-success').html(L("<h4>Rebooting controller ...</h4>")).show()
      $(window).on('serverReady', @onServerReady)
      $(window).on('serverTimeout', @onServerTimeout)
      checker = ServerStateCheckerSingleton.instance()
      checker.check()
      
    onRebootFailure: =>
      @resetUpgradeState()
      @$el.find('.alert').removeClass('alert-success').addClass('alert-error').html(L("<h4>Failed to reboot controller.</h4>")).show()

    doReboot: =>
      @$el.find('.alert').removeClass('alert-error').addClass('alert-success').html(L("<h4>Start to reboot controller ...</h4>")).show()
      
      $.ajax
        type: 'POST'
        url: 'utility.php'
        dataType: 'json'
        data:
          action: 'reboot'
        success: @onRebootSuccess
        error: @onRebootFailure

    doUpgrade: =>
      primaryBtn = @$el.find('.btn-danger')
      return if primaryBtn.hasClass("disabled")

      @removeErrors()
      return unless @validateFileInput()

      @spinner = new Spinner() unless @spinner
      @spinner.spin(@$el[0])

      primaryBtn.addClass("disabled").attr("disabled", "disabled")

      btnText = primaryBtn.html()
      unless _.str.endsWith(btnText, '...')
        primaryBtn.html("#{btnText}...")
      @$el.find('.alert').removeClass('alert-success').removeClass('alert-error').text('').hide()

      @$el.find('form').submit()

      # url = "utility.php"
      # $.ajax
      #   type: "POST"
      #   url: url
      #   data:
      #     'action': 'reboot'
      #   dataType: 'json'
      #   beforeSend: =>
      #     @spinner = new Spinner() unless @spinner
      #     @spinner.spin(@$el[0])
      #     primaryBtn = @$el.find('.btn-danger')
      #     primaryBtn.addClass("disabled").attr("disabled", "disabled")
      #     btnText = primaryBtn.html()
      #     unless _.str.endsWith(btnText, '...')
      #       primaryBtn.html("#{btnText}...")
      #     @$el.find('.alert').removeClass('alert-success').removeClass('alert-error').text('').hide()
      #   success: (data)=>
      #     if data.error
      #       @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
      #     else if data.redirect?
      #       redirect(data.redirect)
      #     else
      #       @$el.find('.alert').addClass('alert-success').html("<h4>Controller is restarting...</h4>").show()
      #   error: (data, status)=>
      #     console.log "error status: " + status
      #     @$el.modal('hide')
      #
      #     view = new AlertMessageView
      #     view.render("Error", status)
      #   complete: (data, status)=>
      #     primaryBtn = @$el.find('.btn-danger')
      #     primaryBtn.removeClass("disabled").removeAttr("disabled")
      #
      #     btnText = primaryBtn.html()
      #     if _.str.endsWith(btnText, '...')
      #       primaryBtn.html(_.str.rtrim(btnText, '.'))
      #     @spinner?.stop()

  class UtilityDateTimeView extends UtilityBaseDialog
    tagName: 'div'
    className: "modal hide fade"
    attributes:
      'data-backdrop': 'static'

    @timezones = null

    initialize: ->
      @spinner = null
      @timestamp = null
      # d = new Date(2000, 0, 1)
      d = new Date(Date.UTC(2000, 0, 1, 0, 0, 0))
      @baseSecsFromEpoch = d.getTime()/1000

      @listenTo(@, 'dataLoaded', @onDataLoaded)

    timeLabel: ->
      now = @timestamp
      _.str.sprintf('%02d-%02d-%02d %02d:%02d:%02d',
        now.getFullYear(), now.getMonth()+1, now.getDate(),
        now.getHours(), now.getMinutes(), now.getSeconds())

    render: ->
      @$el.empty()

      @timestamp = new timezoneJS.Date(Date.now())

      form_html = """
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label" for="inputDateTime">#{L('DateTime:')}</label>
          <div class="controls input-append date" style="display:block">
            <input type="text" id="inputDateTime" value="#{@timeLabel()}">
            <span class='add-on'><i data-time-icon='icon-time' data-date-icon='icon-calendar'> </i></span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label" for="inputTimeZone">#{L('TimeZone:')}</label>
          <div class="controls">
            <select id="inputTimeZone">
            </select>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label" for="inputUTCOffset">#{L('UTC Offset:')}</label>
          <div class="controls">
            <input type="text" id="inputUTCOffset" disabled>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label">#{L('UTC Offset Mode:')}</label>
          <div class="controls">
            <label class="radio inline">
            <input type="radio" name="UTCOffsetMode" value="OS" checked> #{L('OS')}
            </label>
            <label class="radio inline">
            <input type="radio" name="UTCOffsetMode" value="Configuration"> #{L('Configuration')}
            </label>
          </div>
        </div>
      </form>
      """

      html = $("#modal_tpl").html()
      @$el.html(_.template(html, {
        title: L("DateTime <small>change controller's date time or timezone</small>"),
        message: form_html,
        btnHtml: "<a href='#' class='btn btn-primary'>#{L('Ok')}</a>"
      }))

      @populateTZSelect()

      @$el.modal()

      @$el.find(".date").datetimepicker(
        format: "yyyy-MM-dd hh:mm:ss"
        maskInput: true
      ).on('changeDate', @onDateTimeChanged)

      @$el.find('form').submit(@submitHandler)
      @$el.find('.btn-primary').on('click', @doChangeDateTime)

    populateTZSelect: ->
      if UtilityDateTimeView.timezones
        @doPopulateTZSelect()
      else
        timezoneJS.timezone.loadingScheme = timezoneJS.timezone.loadingSchemes.PRELOAD_ALL
        timezoneJS.timezone.zoneFileBasePath = 'tz'
        timezoneJS.timezone.init({callback: @doPopulateTZSelect})

    doPopulateTZSelect: =>
      unless UtilityDateTimeView.timezones
        UtilityDateTimeView.timezones = timezoneJS.timezone.getAllZones()
      tzselect = @$el.find('#inputTimeZone')
      tzselect.append("<option>#{tz}</option>") for tz in UtilityDateTimeView.timezones
      tzselect.change(@onTZChanged)

      @trigger('dataLoaded')

    onDataLoaded: =>
      instance = DateTimeUtilitySingleton.instance()
      instance.refresh(
        success: @updateUI
        error: =>
          @$el.modal('hide')
      )

    updateUI: =>
      instance = DateTimeUtilitySingleton.instance()

      tzselect = @$el.find('#inputTimeZone')
      tz = instance.get('tz').value
      tzselect.val(tz)

      offsetMode = _.str.toBoolean(instance.get('osUtcOffset').value) or 'OS' and 'Configuration'
      @$el.find("input[name='UTCOffsetMode'][value='#{offsetMode}']").prop('checked', true)

      utcOffset = _.str.toNumber(instance.get('utcOffset').value)
      @$el.find('#inputUTCOffset').val("#{utcOffset/(60*60.0)} hours")

      nanos = instance.get('nanos').value
      secs = nanos/1000000000 + @baseSecsFromEpoch
      # if offsetMode == 'Configuration'
      #   secs = secs + utcOffset

      @timestamp = new timezoneJS.Date(secs*1000, 'UTC')

      @onTZChanged()

    onTZChanged: =>
      tzselect = @$el.find('#inputTimeZone')
      newTZ = tzselect.val()
      @timestamp.setTimezone(newTZ)

      @$el.find("#inputDateTime").val(@timeLabel())
      @$el.find("#inputUTCOffset").val("#{-1*@timestamp.getTimezoneOffset()/60.0} hours")

    onDateTimeChanged: (e)=>
      newVal = @$el.find('#inputDateTime').val()
      [date, time] = newVal.split(' ')
      [year,month,day] = _.map(date.split('-'), _.str.toNumber)
      [hour,min,sec] = _.map(time.split(':'), _.str.toNumber)
      @timestamp = new timezoneJS.Date(year, month-1, day, hour, min, sec, @timestamp.getTimezone())

    doChangeDateTime: =>
      tz = @timestamp.getTimezone()
      isOSUTCOffsetMode = @$el.find("input[name=UTCOffsetMode]:checked").val() == 'OS'
      utcOffset = -1*@timestamp.getTimezoneOffset()*60

      # GOTCHA: due to a bug in DateTimeService kit, don't set second
      # sysclockInNs = "#{@timestamp.getTime() - @baseSecsFromEpoch*1000}000000"
      sysclockInNs = "#{@timestamp.getTime() - (@timestamp.getSeconds()+@baseSecsFromEpoch)*1000}000000"

      @spinner = new Spinner() unless @spinner
      @spinner.spin(@$el[0])
      utility = DateTimeUtilitySingleton.instance()
      utility.save(isOSUTCOffsetMode, tz, utcOffset, sysclockInNs,
        complete: =>
          @spinner.stop()
      )

    submitHandler: (e)=>
      e.preventDefault()
      @doChangeDateTime()

  class UtilityServicePortView extends UtilityBaseDialog
    tagName: 'div'
    className: "modal hide fade"
    attributes:
      'data-backdrop': 'static'

    initialize: ->
      @spinner = null

    render: ->
      @$el.empty()

      form_html = """
      <form class="form-horizontal">
        <div class="control-group">
          <label class="control-label" for="inputHttpPort">#{L('Http Port:')}</label>
          <div class="controls">
            <input type="text" id="inputHttpPort" value="">
            <span class='help-inline hide'></span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label" for="inputFtpPort">#{L('Ftp Port:')}</label>
          <div class="controls">
            <input type="text" id="inputFtpPort" value="">
            <span class='help-inline hide'></span>
          </div>
        </div>
        <div class="control-group">
          <button id="AdvPortConfigToggle" class="btn btn-link pull-right" type="button">#{L('Advanced Config')}</button>
        </div>
        <div class="control-group adv-port-config" style="display:none;">
          <label class="control-label" for="input1stFtpDataPort">#{L('Min Ftp Data Port:')}</label>
          <div class="controls">
            <input type="text" id="input1stFtpDataPort" value="">
            <span class='help-inline hide'></span>
          </div>
        </div>
        <div class="control-group adv-port-config" style="display:none;">
          <label class="control-label" for="input2ndFtpDataPort">#{L('Max Ftp Data Port:')}</label>
          <div class="controls">
            <input type="text" id="input2ndFtpDataPort" value="">
            <span class='help-inline hide'></span>
          </div>
        </div>
      </form>
      """

      html = $("#modal_tpl").html()
      @$el.html(_.template(html, {
        title: L("Service Port <small>config controller service's ports</small>"),
        message: form_html,
        btnHtml: "<a href='#' class='btn btn-primary'>#{L('Ok')}</a><a href='#' class='btn btn-info'>#{L('Reset')}</a>"
      }))

      @$el.modal()

      @$el.find('form').submit(@submitHandler)
      @$el.find('.btn-primary').on('click', @doChangeServicePort)
      @$el.find('.btn-info').on('click', @doResetToDefaultePort)
      @$el.find('#AdvPortConfigToggle').on('click', @toggleAdvPortConfig)
      
      @loadCurConfig()
      
    loadCurConfig: ->
      $.ajax
        type: 'GET'
        url: 'service_config.php'
        dataType: 'json'
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
        success: (data)=>
          @onConfigLoaded(data)
        complete: =>
          @spinner.stop()
          
    onConfigLoaded: (data)->
      @$el.find("#inputHttpPort").val(data['http_port'])
      @$el.find("#inputFtpPort").val(data['ftp_port'])
      @$el.find("#input1stFtpDataPort").val(data['1st_ftp_data_port'])
      @$el.find("#input2ndFtpDataPort").val(data['2nd_ftp_data_port'])
      
    renderError: (selector, text)->
      elem = @$el.find(selector)
      elem.siblings('.help-inline').text(text).show()
      elem.parent('.controls').parent('.control-group').addClass('error')

    validatePort: (selector)->
      port = @$el.find(selector).val()

      portNum = _.str.toNumber(port)
      if _.isNaN(portNum) or (portNum <= 0 or portNum >= 65535)
        return false
      return true

    validate: ->
      if not @validatePort('#inputHttpPort')
        @renderError('#inputHttpPort', L("Invalid port value"))
        return false

      if not @validatePort('#inputFtpPort')
        @renderError('#inputFtpPort', L("Invalid port value"))
        return false

      if not @validatePort('#input1stFtpDataPort')
        @renderError('#input1stFtpDataPort', L("Invalid port value"))
        return false

      if not @validatePort('#input2ndFtpDataPort')
        @renderError('#input2ndFtpDataPort', L("Invalid port value"))
        return false
      
      lowerPort = _.str.toNumber(@$el.find('#input1stFtpDataPort').val())
      upperPort = _.str.toNumber(@$el.find('#input2ndFtpDataPort').val())
      if lowerPort >= upperPort
        @renderError('#input2ndFtpDataPort', L("MaxFtpDataPort must be larger than MinFtpDataPort."))
        return false

      return true

    toggleAdvPortConfig: =>
      @$el.find('.adv-port-config').toggle()
      
    doResetToDefaultePort: =>
      @$el.find("#inputHttpPort").val(80)
      @$el.find("#inputFtpPort").val(21)
      @$el.find("#input1stFtpDataPort").val(40000)
      @$el.find("#input2ndFtpDataPort").val(50000)

    doChangeServicePort: =>
      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')

      if not @validate()
        return
      
      formValues =
        'ftp_port': @$el.find('#inputFtpPort').val()
        'http_port': @$el.find('#inputHttpPort').val()
        '1st_ftp_data_port': @$el.find('#input1stFtpDataPort').val()
        '2nd_ftp_data_port': @$el.find('#input2ndFtpDataPort').val()
      
      $.ajax
        url: 'service_config.php'
        type: 'POST'
        dataType: 'json'
        data: formValues
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
        success: (data)=>
          @$el.find('.alert').addClass('alert-success').html(L("<h4>Succeed config service's ports!</h4>reloading now, please wait ...")).show()
          setTimeout(
            =>
              @$el.modal('hide')
              location.reload()
            ,3500)
        complete: =>
          @spinner.stop()

    submitHandler: (e)=>
      e.preventDefault()
      @doChangeServicePort()

  class UtilityChangeSysPasswordView extends UtilityBaseDialog
    tagName: 'div'
    className: "modal hide fade"
    attributes:
      'data-backdrop': 'static'

    initialize: ->
      @spinner = null

    render: ->
      @$el.empty()

      form_html = """
      <form class="form-horizontal" autocomplete="nope">
        <div class="control-group">
          <label class="control-label" for="inputOldPass">#{L('Old Password:')}</label>
          <div class="controls">
            <input type="password" id="inputOldPass" value="" minlength=6 maxlength=32>
            <span class='help-inline hide'></span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label" for="inputNewPass">#{L('New Password:')}</label>
          <div class="controls">
            <input type="password" id="inputNewPass" value="" minlength=6 maxlength=32>
            <span class='help-inline hide'></span>
          </div>
        </div>
        <div class="control-group">
          <label class="control-label" for="inputNewPass2">#{L('Confirm Password:')}</label>
          <div class="controls">
            <input type="password" id="inputNewPass2" value="" password minlength=6 maxlength=32>
            <span class='help-inline hide'></span>
          </div>
        </div>
      </form>
      """

      html = $("#modal_tpl").html()
      @$el.html(_.template(html, {
        title: L("Change OS Password <small>change OS account's password</small>"),
        message: form_html,
        btnHtml: "<a href='#' class='btn btn-primary'>#{L('Ok')}</a>"
      }))

      @$el.modal()

      @$el.find('form').submit(@submitHandler)
      @$el.find('.btn-primary').on('click', @doChangeSysPassword)
      
    renderError: (selector, text)->
      elem = @$el.find(selector)
      elem.siblings('.help-inline').text(text).show()
      elem.parent('.controls').parent('.control-group').addClass('error')

    validatePass: (selector)->
      val = $(selector).val()
      if not /^[!-~]{6,32}$/.test(val)
        @renderError(selector, L("password's length must between 6-32 characters"))
        return false

      # must include lowercase character
      if not /[a-z]/.test(val)
        @renderError(selector, L("password must include lowercase characters"))
        return false
      
      # must include uppercase character
      if not /[A-Z]/.test(val)
        @renderError(selector, L("password must include uppercase characters"))
        return false
      
      # must include number
      if not /[0-9]/.test(val)
        @renderError(selector, L("password must include number characters"))
        return false

      # must include punctuation 
      if not /[!-/:-@^-`{-~]/.test(val)
        @renderError(selector, L("password must include punctuation characters"))
        return false
      
      return true

    validate: ->
      old_pass = $('#inputOldPass').val()
      if not /^[!-~]{6,32}$/.test(old_pass)
        @renderError('#inputOldPass', L("password's length must between 6-32 characters"))
        return false
      
      new_pass = $('#inputNewPass').val()
      if not @validatePass('#inputNewPass')
        return false
      
      new_pass2 = $('#inputNewPass2').val()
      if new_pass != new_pass2
        @renderError('#inputNewPass2', L("confirmed password does not match new password"))
        return false

      if not @validatePass('#inputNewPass2')
        return false
      
      return true

    doChangeSysPassword: =>
      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')

      if not @validate()
        return
      
      formValues =
        'action': 'changePasswd'
        'account':
          'old_password': @$el.find('#inputOldPass').val()
          'new_password': @$el.find('#inputNewPass').val()

      $.ajax
        url: 'os_account_management.php'
        type: 'POST'
        dataType: 'json'
        data: formValues
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
        success: (data)=>
          if data.error
            @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
          else
            @$el.find('.alert').addClass('alert-success').html(L("<h4>Password is changed successfully.</h4>")).show()
            setTimeout(
              =>
                @$el.modal('hide')
              ,1000)
        complete: =>
          @spinner.stop()
      
    submitHandler: (e)=>
      e.preventDefault()
      @doChangeSysPassword()

  class UtilityToolsView extends Backbone.View
    el: '#utilityTools'

    events:
      'click #utilityBackup': 'backup'
      'click #utilityRestore': 'restore'

      'click #utilityRestart': 'restart'
      'click #utilityReboot': 'reboot'

      'click #utilityUpgradeFirmware': 'upgradeFirmware'

      'click #utilityDateTime': 'changeDateTime'
      'click #utilityServicePort': 'changeServicePort'
      
      'click #utilityChangeSysPassword': 'changeSysPassword'

    initialize: ->
      @backupView = null
      @restoreView = null
      @restartView = null
      @rebootView = null
      @upgradeView = null
      @dateTimeView = null
      @servicePortView = null

    backup: ->
      @backupView = new UtilityBackupView unless @backupView
      @backupView.render()

    restore: ->
      @restoreView = new UtilityRestoreView unless @restoreView
      @restoreView.render()

    restart: ->
      @restartView = new UtilityRestartView unless @restartView
      @restartView.render()

    reboot: ->
      @rebootView = new UtilityRebootView unless @rebootView
      @rebootView.render()

    upgradeFirmware: ->
      @upgradeView = new UtilityUpgradeView unless @upgradeView
      @upgradeView.render()

    changeDateTime: ->
      @dateTimeView = new UtilityDateTimeView unless @dateTimeView
      @dateTimeView.render()
      
    changeServicePort: ->
      @servicePortView = new UtilityServicePortView unless @servicePortView
      @servicePortView.render()
      
    changeSysPassword: ->
      @changeSysPasswordView = new UtilityChangeSysPasswordView unless @changeSysPasswordView
      @changeSysPasswordView.render()

  class AccountActionsView extends Backbone.View
    el: '#userBtn'

    events:
      'click #userProfile': 'updateProfile'
      'click #userManagement': 'manageUser'

    initialize: ->
      @passwordView = null
      @accountManagementView = null

    updateProfile: ->
      unless @passwordView
        @passwordView = new ChangePasswordModalView
      @passwordView.render()

    manageUser: ->
      unless @accountManagementView
        @accountManagementView = new AccountManagementView
      @accountManagementView.render()


  class AccountManagementView extends Backbone.View
    el: '#accountManagementModal'

    render: ->
      unless @permissionsView?
        @permissionsView = new PermissionsView() 
        @.listenTo(@permissionsView, 'userSwitched', @onUserSwitched)
      @permissionsView.render()

      unless @accountsView?
        @accountsView = new AccountsView()
        @accountsView.on('accountDeleted', @permissionsView.onUserDeleted)
        @.listenTo(@accountsView, 'userSwitched', @onUserSwitched)
      @accountsView.render()

      unless @newAccountView?
        @newAccountView = new NewAccountView()
        @newAccountView.on('newAccountCreated', @permissionsView.render)
        @newAccountView.on('newAccountCreated', @accountsView.render)
      @newAccountView.render()

      @sessionTTLView = new SessionTTLView() unless @sessionTTLView?
      @sessionTTLView.render()
      
      @authKeyView = new AuthKeyView() unless @authKeyView?
      @authKeyView.render()

      @$el.modal()

    onUserSwitched: (user_id)=>
      @accountsView.switchUser(user_id) if @accountsView?
      @permissionsView.switchUser(user_id) if @permissionsView?

  class ChangePasswordModalView extends Backbone.View
    el: "#changePasswordModal"

    events:
      'click .btn-primary': 'apply'

    initialize: ->
      @$el.find('form').submit(@submitHandler)

    render: ->
      @$el.find('input').val('')
      @$el.find('.alert').hide()
      @$el.find('.help-inline').val('').hide()
      @$el.find('.control-group').removeClass('error')
      @$el.modal()

    oldPassword: ->
      @$el.find('#inputOldPassword').val()

    newPassword: ->
      @$el.find('#inputNewPassword').val()

    confirmedPassword: ->
      @$el.find('#inputConfirmPassword').val()

    renderError: (selector, text)->
      elem = @$el.find(selector)
      elem.siblings('.help-inline').text(text).show()
      elem.parent('.controls').parent('.control-group').addClass('error')

    validate: ->
      if _.str.isBlank(@oldPassword())
        @renderError("#inputOldPassword", L("please input your current password"))
        return false

      error = validateNormalText(@newPassword(), 'password', 8)
      if error
        @renderError("#inputNewPassword", error)
        return false

      if _.str.isBlank(@confirmedPassword())
        @renderError("#inputConfirmPassword", L("please confirm your new password"))
        return false

      if @newPassword() != @confirmedPassword()
        @$el.find('.alert').addClass('alert-error').text(L("new passwords do not match")).show()
        return false

      return true
    
    doSendRequest: (authToken)->
      url = "profile.php"

      oldPass = @oldPassword()
      newPass = @newPassword()

      [token1, token2] = _.str.words(authToken, '_')
      shaObj1 = new jsSHA(oldPass + token1, 'TEXT')
      shaObj2 = new jsSHA(shaObj1.getHash('SHA-1', 'HEX') + token2, 'TEXT')
      shaObj3 = new jsSHA(newPass + token1, 'TEXT')
      
      # formValues =
      #   'user[old_password]': 
      #   'user[new_password]': 
      formValues =
        'user[old_auth_hash]': shaObj2.getHash('SHA-1', 'HEX')
        'user[new_auth_hash]': shaObj3.getHash('SHA-1', 'HEX')

      $.ajax
        url: url
        type: "POST"
        dataType: 'json'
        data: formValues
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled")
          @$el.find('.btn-primary').attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            console.info 'password is updated'
            @$el.modal('hide')
        complete: =>
          @$el.find('.btn-primary').removeClass("disabled")
          @$el.find('.btn-primary').removeAttr("disabled")
    
    sendRequest: ->
      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')

      if not @validate()
        return
      
      $.ajax
        url: 'signin.php'
        type: 'GET'
        data:
          'user[name]': curUserName()
        dataType: 'json'
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled")
          @$el.find('.btn-primary').attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            console.info 'authtoken is fetched'
            @doSendRequest(data.authToken)
      .always (data, status, jqXHR)=>
        if status != 'success' or data.error
          @$el.find('.btn-primary').removeClass("disabled")
          @$el.find('.btn-primary').removeAttr("disabled")

    apply: =>
      @sendRequest()

    submitHandler: (e)=>
      e.preventDefault()
      @apply()

  class PermissionView extends Backbone.View
    model: UserPermission

    events:
      'click .btn-primary': 'apply'

    initialize: (user, container, insertMode='append')->
      @model = user
      @container = container
      @insertMode = insertMode

    render: ->
      user_list_container = @container.find('.users-list')
      user_permissions_container = @container.find('.users-permissions')

      index = _.indexOf(@model.collection.models, @model)
      if @insertMode == 'append'
        user_list_container.append(@renderUser(@model, index))
        user_permissions_container.append(@renderPermissions(@model, index))
      else
        user_list_container.prepend(@renderUser(@model, index))
        user_permissions_container.prepend(@renderPermissions(@model, index))

      @setElement($("#user_#{@model.get('user_id')}"))
      @

    renderUser: (user, index)->
      html = $("#user_name_tpl").html()
      _.template(html, {user: user, index: index})

    renderPermissions: (user, index)->
      html = $("#user_permissions_tpl").html()
      _.template(html, {user: user, index: index})

    syncData: ->
      home_page = @$el.find('input.home_page-radio:checked').val()
      @model.set('home_page', home_page)

      @$el.find('tbody tr').each (index, elem)=>
        tr = $(elem)
        path = tr.data("grpath")
        permStr = tr.find('.perm-radio:checked').val()
        @model.updateData(path, permStr)
        
      @model.updateDevPerm(@$el.find('.dev-readable-checkbox').is(":checked"),
        @$el.find('.dev-writable-checkbox').is(":checked"))
        
      $.ajax
        url: @model.url()
        type: "POST"
        dataType: 'json'
        data: @model.postData()
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled")
          @$el.find('.btn-primary').attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').first().removeClass('alert-success').addClass('alert-error').text(data.error.text).slideDown(300)
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').first().removeClass('alert-error').addClass('alert-success').text(L('Permissions are updated')).slideDown(300)
            @updateHomeIcon()
            console.info 'permissions are updated'
        complete: =>
          @$el.find('.btn-primary').removeClass("disabled")
          @$el.find('.btn-primary').removeAttr("disabled")

    apply: =>
      @syncData()

    updateHomeIcon: ->
      @$el.find('i.icon-home').removeClass('icon-home').addClass('icon-picture')
      tr = @$el.find('input.home_page-radio:checked').parents('tr')
      tr.find('i.icon-picture').removeClass('icon-picture').addClass('icon-home')

  class PermissionsView extends Backbone.View
    el: '#permissionsPanel'

    initialize: ->
      @model = UserPermissionsSingleton.instance()
      @subViews = []
      @listenTo(@model, "add", @onUserAdded)
      @listenTo(@model, "sync", @renderSubViews)

    render: =>
      @$el.empty()
      $("""
          <div class="tabbable tabs-left" id="permissions-content">
            <ul class="nav nav-tabs users-list" ></ul>
            <div class="tab-content users-permissions" ></div>
          </div>
      """).appendTo(@$el)
      @model.fetch({error: @handleDataLoadError})

    renderSubViews: ->
      @subViews = _.sortBy(@subViews, (r)->
        parseInt(r.model.get('user_id'))
      )
      for view in @subViews
        view.render()
      @switchUser(1)
      @$el.find('.dev-perm-toggle').on('click', @toggleDevPermConfig)
      @$el.find("a[data-toggle='tab']").on('shown', @onTabChanged)

    onUserAdded: (user)->
      subView = new PermissionView(user, @$el)
      @subViews.push(subView)
      @$el.find("a[data-toggle='tab']").off('shown').on('shown', @onTabChanged)

    onUserDeleted: (user_id)=>
      headers = @$el.find('.nav-tabs')
      contents = @$el.find('tab-content')

      li = headers.find("a[href='#user_#{user_id}']").closest('li')
      li.siblings().first().find('a').tab('show')

      li.remove()
      contents.find("#user_#{user_id}").remove()

      for subView, index in @subViews
        continue if !subView or user_id != subView.model.get('user_id')

        @model.remove(subView.model)
        @subViews.splice(index, 1)

    handleDataLoadError: (error)->
      console.error error

    toggleDevPermConfig: =>
      @$el.find('.dev-perm-config').toggle()

    onTabChanged: (e)=>
      @.trigger("userSwitched", $(e.target).data('user_id'))

    switchUser: (user_id)->
      cur_user_id = @$el.find(".nav-tabs li.active").find("a[data-toggle='tab']").data('user_id')
      if (cur_user_id != user_id)
        @$el.find(".nav-tabs a[href='#user_#{user_id}']").tab('show')

  class AccountView extends Backbone.View
    model: Account

    events:
      'click .btn-primary': 'apply'
      'click .delete-account-btn': 'deleteAccount'

    initialize: (account, container, insertMode='append')->
      @model = account
      @container = container
      @insertMode = insertMode

    render: ->
      accounts_list_container = @container.find('.accounts-list')
      accounts_details_container = @container.find('.accounts-details')

      index = _.indexOf(@model.collection.models, @model)
      if @insertMode == 'append'
        accounts_list_container.append(@renderAccount(@model, index))
        accounts_details_container.append(@renderDetails(@model, index))
      else
        accounts_list_container.prepend(@renderAccount(@model, index))
        accounts_details_container.prepend(@renderDetails(@model, index))

      @setElement($("#account_#{@model.get('user_id')}"))

      @$el.find('.inputEnableUtility').on('click', @updateUI)
      @updateUI()
      @

    updateUI: =>
      utilityCheckBox = @$el.find('.inputEnableUtility')
      systemCheckBox = @$el.find('.inputEnableSystem')
      if utilityCheckBox.is(':checked')
        systemCheckBox.removeAttr('disabled')
      else
        systemCheckBox.attr('checked', false)
        systemCheckBox.attr('disabled', true)

    renderAccount: (account, index)->
      html = $("#account_name_tpl").html()
      _.template(html, {account: account, index: index})

    renderDetails: (account, index)->
      html = $("#account_details_tpl").html()
      _.template(html, {account: account, index: index})

    renderError: (selector, text)->
      elem = @$el.find(selector)
      elem.siblings('.help-inline').text(text).show()
      elem.parent('.controls').parent('.control-group').addClass('error')

    validate: ->
      password = @$el.find('#inputPasswordUpdate').val()
      return true if _.str.isBlank(password)

      error = validateNormalText(password, 'password', 8)
      if error
        @renderError('#inputPasswordUpdate', error)
        return false

      return true
    
    doSyncData: (authToken)=>
      formElem = @$el.find('form')
      @model.enableUtility(formElem.find('.inputEnableUtility').is(':checked'))
      @model.enableSystem(formElem.find('.inputEnableSystem').is(':checked'))
      @model.enableAccountManagement(formElem.find('.inputEnableAccountManagement').is(':checked'))
      @model.enablePasswordChange(formElem.find('.inputEnablePasswordChange').is(':checked'))
      @model.enableDashboard(formElem.find('.inputEnableDashboard').is(':checked'))
      @model.enableDashboardLanding(formElem.find('.inputDashboardAsLandingPage').is(':checked'))

      data =
        action: 'updateAccount'
        user: @model.postData()
      newPassword = formElem.find('#inputPasswordUpdate').val()
      unless _.str.isBlank(newPassword)
        unless authToken
          console.error("invalid authToken")
          return

        [token1, token2] = _.str.words(authToken, '_')
        shaObj1 = new jsSHA(newPassword + token1, 'TEXT')
        data['user']['authHash'] = shaObj1.getHash('SHA-1', 'HEX')

      $.ajax
        url: @model.url()
        type: "POST"
        dataType: 'json'
        data: data
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled")
          @$el.find('.btn-primary').attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').removeClass('alert-success').addClass('alert-error').text(data.error.text).slideDown(300)
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').removeClass('alert-error').addClass('alert-success').text(L('Account is updated')).slideDown(300)
            console.info 'account is updated'
        complete: =>
          @$el.find('.btn-primary').removeClass("disabled")
          @$el.find('.btn-primary').removeAttr("disabled")

    syncData: ->
      formElem = @$el.find('form')
      newPassword = formElem.find('#inputPasswordUpdate').val()
      if _.str.isBlank(newPassword)
        @doSyncData()
      else
        $.ajax
          url: 'signin.php'
          dataType: 'json'
          data:
            'user[name]': @model.id
          beforeSend: =>
            @$el.find('.btn-primary').addClass("disabled")
            @$el.find('.btn-primary').attr("disabled", "disabled")
          success: (data)=>
            if data.error
              @$el.find('.alert').removeClass('alert-success').addClass('alert-error').text(data.error.text).slideDown(300)
            else if data.redirect?
              redirect(data.redirect)
            else
              @doSyncData(data.authToken)
        .always (data, status, jqXHR)=>
          if status != 'success' or data.error
            @$el.find('.btn-primary').removeClass("disabled")
            @$el.find('.btn-primary').removeAttr("disabled")

    apply: =>
      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')

      @syncData() if @validate()

    deleteAccount: =>
      confirmed = confirm(L("Are you sure to delete account '{name}' ?").supplant(name: @model.get('name')))
      return unless confirmed

      user_id = @model.get('user_id')
      formValues =
        'action': 'deleteAccount'

      delBtn = @$el.find('.delete-account-btn')

      $.ajax
        url: @model.url()
        type: "POST"
        dataType: 'json'
        data: formValues
        beforeSend: =>
          delBtn.addClass("disabled")
          delBtn.attr("disabled", "disabled")
        success: (data)=>
          alertElem = @$el.find('.alert')
          if data.error
            alertElem.removeClass('alert-success').addClass('alert-error').text(data.error.text).slideDown(300)
          else if data.redirect?
            redirect(data.redirect)
          else
            alertElem.removeClass('alert-error').addClass('alert-success').text(L('Account is deleted')).slideDown(300)
            @trigger("accountDeleted", @, user_id)
        complete: =>
          delBtn.removeClass("disabled")
          delBtn.removeAttr("disabled")

  class AccountsView extends Backbone.View
    el: '#accountsPanel'

    initialize: ->
      @model = AccountsSingleton.instance()
      @subViews = []
      @listenTo(@model, "add", @onAccountAdded)
      @listenTo(@model, "sync", @renderSubViews)

    render: =>
      @$el.empty()
      $("""
          <div class="tabbable tabs-left" id="accounts-content">
            <ul class="nav nav-tabs accounts-list" ></ul>
            <div class="tab-content accounts-details" ></div>
          </div>
      """).appendTo(@$el)
      @model.fetch({error: @handleDataLoadError})

    handleDataLoadError: (error)->
      console.error error

    renderSubViews: ->
      @subViews = _.sortBy(@subViews, (r)->
        parseInt(r.model.get('user_id'))
      )
      for view in @subViews
        view.render()

      @switchUser(1)
      @$el.find("a[data-toggle='tab']").on('shown', @onTabChanged)

    onAccountAdded: (account)->
      subView = new AccountView(account, @$el)
      @subViews.push(subView)
      @listenTo(subView, 'accountDeleted', @onAccountDeleted)
      @$el.find("a[data-toggle='tab']").off('shown').on('shown', @onTabChanged)

    onAccountDeleted: (subView, user_id)=>
      headers = @$el.find('.nav-tabs')
      contents = @$el.find('tab-content')

      li = headers.find("a[href='#account_#{user_id}']").closest('li')
      li.siblings().first().find('a').tab('show')

      li.remove()
      contents.find("#account_#{user_id}").remove()

      index = @model.remove(subView.model)
      index = @subViews.indexOf(subView)
      if (index > -1)
        @subViews.splice(index, 1)

      @trigger("accountDeleted", user_id)

    onTabChanged: (e)=>
      @.trigger("userSwitched", $(e.target).data('user_id'))

    switchUser: (user_id)->
      cur_user_id = @$el.find(".nav-tabs li.active").find("a[data-toggle='tab']").data('user_id')
      if (cur_user_id != user_id)
        @$el.find(".nav-tabs a[href='#account_#{user_id}']").tab('show')

  class NewAccountView extends Backbone.View
    el: '#newAccountPanel'

    render: ->
      html = $("#new_account_tpl").html()
      @$el.html(_.template(html, {}))
      @$el.find('form').submit(@submitHandler)

      @$el.find('.inputEnableUtility').on('click', @updateUI)
      @updateUI()
      @

    submitHandler: (event)=>
      event.preventDefault()
      @createAccount()

    updateUI: =>
      utilityCheckBox = @$el.find('.inputEnableUtility')
      systemCheckBox = @$el.find('.inputEnableSystem')
      if utilityCheckBox.is(':checked')
        systemCheckBox.removeAttr('disabled')
      else
        systemCheckBox.attr('checked', false)
        systemCheckBox.attr('disabled', true)

    userName: ->
      @$el.find('#inputName').val()

    password: ->
      @$el.find('#inputPassword').val()

    isUtilityEnabled: ->
      @$el.find('.inputEnableUtility').is(':checked')

    isSystemEnabled: ->
      @$el.find('.inputEnableSystem').is(':checked')
      
    isAccountManagementEnabled: ->
      @$el.find('.inputEnableAccountManagement').is(':checked')
      
    isPasswordChangeEnabled: ->
      @$el.find('.inputEnablePasswordChange').is(':checked')
      
    isDashboardEnabled: ->
      @$el.find('.inputEnableDashboard').is(':checked')
      
    isDashboardLanding: ->
      @$el.find('.inputDashboardAsLandingPage').is(':checked')

    renderError: (selector, text)->
      elem = @$el.find(selector)
      elem.siblings('.help-inline').text(text).show()
      elem.parent('.controls').parent('.control-group').addClass('error')

    validate: ->
      error = validateNormalText(@userName(), 'name', 2)
      if error
        @renderError('#inputName', error)
        return false

      error = validateNormalText(@password(), 'password', 8)
      if error
        @renderError('#inputPassword', error)
        return false

      return true

    createAccount: ->
      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')

      if not @validate()
        return

      url = "account_management.php"

      token = randStr()
      shaObj = new jsSHA(@password() + token, 'TEXT')
      authHash = shaObj.getHash('SHA-1', 'HEX')
      formValues =
        'action': 'createAccount'
        'user[name]': @userName()
        'user[token]': token
        'user[authHash]': authHash
        'user[utility_enabled]': @isUtilityEnabled() and 't' or 'f'
        'user[system_enabled]': @isSystemEnabled() and 't' or 'f'
        'user[account_management_enabled]': @isAccountManagementEnabled() and 't' or 'f'
        'user[password_change_enabled]': @isPasswordChangeEnabled() and 't' or 'f'
        'user[dashboard_enabled]': @isDashboardEnabled() and 't' or 'f'
        'user[dashboard_as_landing_page]': @isDashboardLanding() and 't' or 'f'

      $.ajax
        url: url
        type: "POST"
        dataType: 'json'
        data: formValues
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled")
          @$el.find('.btn-primary').attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').removeClass('alert-success').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').removeClass('alert-error').addClass('alert-success').text(L('Account is created')).slideDown(300)
            console.info 'Account is created'
            @trigger("newAccountCreated")
            # @$el.modal('hide')
        complete: =>
          @$el.find('.btn-primary').removeClass("disabled")
          @$el.find('.btn-primary').removeAttr("disabled")

  class SessionTTLView extends Backbone.View
    el: '#sessionTTLPanel'

    initialize: ->
      @model = SettingsSingleton.instance()
      @listenTo(@model, "sync", @onSettingsUpdated)

    render: ->
      html = $("#session_ttl_tpl").html()
      @$el.html(_.template(html, {}))
      @$el.find('form').submit(@submitHandler)

      @$el.find('.inputEnableSessionTTL').on('click', @updateUI)
      @model.fetch({error: @handleDataLoadError})
      @

    handleDataLoadError: (error)->
      console.error error
      
    onSettingsUpdated: =>
      @$el.find('.inputEnableSessionTTL').attr("checked", @model.ttl_enabled())
      @$el.find('#inputSessionTTL').val(@model.ttl_value()) if @model.ttl_enabled()
      @updateUI()

    submitHandler: (event)=>
      event.preventDefault()
      @saveSessionTTL()
      
    isSessionTTLEnabled: ->
      sessionCheckBox = @$el.find('.inputEnableSessionTTL')
      return sessionCheckBox.is(':checked')

    updateUI: =>
      if @isSessionTTLEnabled()
        # @$el.find("button").removeClass("disabled").removeAttr("disabled")
        @$el.find("input").removeAttr("disabled")
      else
        # @$el.find("button").addClass("disabled").attr("disabled", "disabled")
        @$el.find("input").attr("disabled", "disabled")

      @$el.find('.inputEnableSessionTTL').removeAttr("disabled")

    renderError: (selector, text)->
      elem = @$el.find(selector)
      elem.siblings('.help-inline').text(text).show()
      elem.parent('.controls').parent('.control-group').addClass('error')

    validate: ->
      strVal = @$el.find('#inputSessionTTL').val()
      intVal = _.str.toNumber(strVal)
      error = L("input is not a valid number") if _.isNaN(intVal)
      error = L("session can not live longer than 1 month") if !error and intVal > 43200
      error = L("input number must > 0") if !error and intVal <= 0

      if error
        @renderError('#inputSessionTTL', error)
        return false

      return true

    saveSessionTTL: ->
      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')
      
      if @isSessionTTLEnabled() and not @validate()
        return

      url = "settings_controller.php"
      formValues =
        'action': 'updateSessionTTL'
        'ttl_enabled': @isSessionTTLEnabled() and 't' or 'f'

      if @isSessionTTLEnabled()
        formValues['ttl_value'] = _.str.toNumber(@$el.find('#inputSessionTTL').val())

      $.ajax
        url: url
        type: "POST"
        dataType: 'json'
        data: formValues
        beforeSend: =>
          @$el.find('.btn-primary').addClass("disabled")
          @$el.find('.btn-primary').attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').removeClass('alert-success').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').removeClass('alert-error').addClass('alert-success').text(L('Settings saved')).slideDown(300)
        complete: =>
          @$el.find('.btn-primary').removeClass("disabled")
          @$el.find('.btn-primary').removeAttr("disabled")
          
  class AuthKeyView extends Backbone.View
    el: '#authKeyPanel'

    initialize: ->
      @model = AuthKeysSingleton.instance()
      @spinner = null
      @listenTo(@model, "sync", @onAuthKeyUpdated)

    render: ->
      html = $("#auth_keys_tpl").html()
      @$el.html(_.template(html, {}))
      @$el.find('form.new-auth-key-form').submit(@submitHandler)

      startMoment = moment()
      picker = @$el.find("#authKeyExpiration").datetimepicker(
        format: "yyyy-MM-dd"
        pickTime: false
        maskInput: true
        # startDate: startMoment.add(1, 'd').toDate()
      ).data('datetimepicker')#.on('changeDate', @onDateTimeChanged)
      picker?.setLocalDate(startMoment.add(1, 'y').toDate())

      @model.fetch({error: @handleDataLoadError})
      @
      

    handleDataLoadError: (error)=>
      console.error error
      
    onAuthKeyUpdated: =>
      @syncUI()

    submitHandler: (event)=>
      event.preventDefault()
      @createAuthKey()
      
    syncUI: =>
      tmpl = $('#auth_key_entry_tpl').html()
      edit_tmpl = $('#auth_key_edit_entry_tpl').html()
      elem = @$el.find("tbody").empty()
      _.each @model.auth_keys(), (auth_key)=>
        auth_key.expired_cls = if moment.utc(auth_key.expired_at) <= moment().utc() then "warning" else ""
        $(_.template(tmpl, auth_key)).appendTo(elem).show(0)
          .find('.del-auth-key').click (event)=>
            @onDeleteAuthKey(event)
          .end().find('.edit-auth-key').click (event)=>
            $(event.target).parents('tr').next('tr').toggle()
        $(_.template(edit_tmpl, auth_key)).appendTo(elem)
          .find('form.edit-auth-key-form').css
            'margin-bottom': 0
            'margin-left': '28px'
          .submit(_.bind(@onEditAuthKey, @))
          .find('.cancel-btn').click (event)=>
            event.preventDefault()
            $(event.target).parents('tr').hide()

      elem.find('.expiry-date-edit-container').datetimepicker
        format: 'yyyy-MM-dd'
        pickTime: false
        maskInput: true

    validate: ->
      true

    renderError: (selector, text)->
      elem = @$el.find(selector)
      elem.siblings('.help-inline').text(text).show()
      elem.parent('.controls').parent('.control-group').addClass('error')
      
    note: ->
      @$el.find('#inputAuthKeyNote').val()
      
    expired_at: ->
      @$el.find('#inputAuthKeyExpiration').val()
      
    onDeleteAuthKey: (event)=>
      confirmed = confirm(L("Are you sure to delete this AuthKey ?"))
      return unless confirmed

      @deleteAuthKey($(event.target).parents('tr'))
      
    onEditAuthKey: (event)=>
      event.preventDefault()
      @editAuthKey($(event.target))

    createAuthKey: ->
      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')
       
      if not @validate()
        return
    
      formValues =
        'action': 'create'
        'note': @note()
        'expired_at': @expired_at()

      $.ajax
        url: @model.url()
        type: "POST"
        dataType: 'json'
        data: formValues
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
          @$el.find('.btn-primary').addClass("disabled").attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').removeClass('alert-success').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').removeClass('alert-error').addClass('alert-success').text(L('Auth key created')).slideDown(300)
            @model.addKey(
              'note': @note()
              'expired_at': @expired_at()
              'key': data.new_key
            )
            @syncUI()
        complete: =>
          @spinner.stop()
          @$el.find('.btn-primary').removeClass("disabled").removeAttr("disabled")
          
    deleteAuthKey: (keyElem)->
      key = keyElem.data('auth_key')
      return unless key?

      btnElem = keyElem.find('.btn')

      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')

      formValues =
        'action': 'delete'
        'key': key

      $.ajax
        url: @model.url()
        type: "POST"
        dataType: 'json'
        data: formValues
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el.find('form')[0])
          btnElem.addClass("disabled").attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').removeClass('alert-success').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').removeClass('alert-error').addClass('alert-success').text(L('Auth key deleted')).slideDown(300)
            @model.deleteKey(key)
            @syncUI()
        complete: =>
          @spinner.stop()
          btnElem.removeClass("disabled").removeAttr("disabled")
          
    editAuthKey: (formElem)->
      @$el.find('.alert').hide()
      @$el.find('.help-inline').hide()
      @$el.find('.control-group').removeClass('error')
       
      key = formElem.parents('tr').data('auth_key')
      expired_at = formElem.find('.expiry-date-edit').val()
      note = formElem.find('.note-edit').val()
      formValues =
        'action': 'update'
        'key': key
        'note': note
        'expired_at': expired_at

      $.ajax
        url: @model.url()
        type: "POST"
        dataType: 'json'
        data: formValues
        beforeSend: =>
          @spinner = new Spinner() unless @spinner
          @spinner.spin(@$el[0])
          formElem.find('.btn-primary').addClass("disabled").attr("disabled", "disabled")
        success: (data)=>
          if data.error
            @$el.find('.alert').removeClass('alert-success').addClass('alert-error').text(data.error.text).show()
          else if data.redirect?
            redirect(data.redirect)
          else
            @$el.find('.alert').removeClass('alert-error').addClass('alert-success').text(L('Auth key updated')).slideDown(300)
            @model.updateKey(
              'key': key
              'note': note
              'expired_at': expired_at
            )
            @syncUI()
        complete: =>
          @spinner.stop()
          formElem.find('.btn-primary').removeClass("disabled").removeAttr("disabled")


  # CPT performance timing data

  class CptTimingData
    constructor: ->

    logLoadingTime: ->
      totalLoadTime = (@canvasRendered - window.performance?.timing?.navigationStart)/1000.0
      domLoadedTime = (window.performance?.timing?.domContentLoadedEventEnd - window.performance?.timing?.navigationStart)/1000.0

      grNavLoadTime = (@grNavDataLoadEnd - @grNavDataLoadStart)/1000.0
      grNavRenderTime = (@grNavRendered - @grNavDataLoadEnd)/1000.0
      grDocLoadTime = (@grDocDataLoadEnd - @grDocDataLoadStart)/1000.0
      bindingDataLoadTime = (@bindingDataLoadEnd - @bindingDataLoadStart)/1000.0
      canvasRenderTime = (@canvasRendered - @bindingDataLoadEnd)/1000.0

      console.info("CPT Graphic Page loaded in #{totalLoadTime}s (DomLoadTime: #{domLoadedTime}s, GrNavLoadTime: #{grNavLoadTime}s, GrNavRenderTime: #{grNavRenderTime}s, GrDocLoadTime: #{grDocLoadTime}s, BindingDataLoadTime: #{bindingDataLoadTime}s, CanvasRenderTime: #{canvasRenderTime}s)")

  window.cpt_timing_data = new CptTimingData

  window.grNavView = new GrNavView
  window.grCanvasView = new GrCanvasView
  window.grNavInstance = GrNavInstance.instance()
  window.compsPoolInstance = ComponentsPool.instance()

  window.accountActionsView = new AccountActionsView
  window.utilityToolsView = new UtilityToolsView
  
# vim: sw=2 ts=2 expandtab
