<?php
//vim: ts=2 sw=2

include_once "base_controller.php";
include_once "data_formatters.php"; 

class SqlQueryController extends BaseController {
  protected function signinRequired() {
    return true;
  }

  protected function authKeyRequired() {
    return true;
  }

  protected function sameHostAuthFree() {
    return true;
  }
  
  public function doGet() {
    $this->handleRequest();
  }
  
  public function doAjaxGet() {
    $this->handleRequest();
  }
  
  protected function tz() {
    $dt_tz = null;
    if (isset($_GET['dt_tz']))
      $dt_tz = $_GET['dt_tz'];
    try {
      $result = new DateTimeZone($dt_tz);
      return $result;
    } catch(Exception $e) {
      $error = L('"dt_tz" is not valid: ') . $dt_tz;
      error_log($error);
      // $default_tz = date_default_timezone_get();
      $default_tz = 'UTC';
      try {
        $result = new DateTimeZone($default_tz);
        return $result;
      } catch(Exception $e) {
        $error = 'default timezone "' . $default_tz .'" is not valid';
        error_log($error);
        $this->renderErrorAndHelp($error);
      }
    }
  }

  protected function handleRequest() {
    $response = array();
    $query = $this->getInputData('sql');
    if (empty($query))
      $this->renderErrorAndHelp(L("'sql' parameter is required!"));

    $u = $this->curUser();
    //only user with admin role can run queries to modify DB
    if (isset($u) && !$u->isAdmin()) {
      if (!startsWith(strtolower(ltrim($query)), "select"))
        $this->renderErrorAndHelp(L("invalid sql query"));
    }

    if (isset($_GET['responseFormat']))
      $format = $_GET['responseFormat'];
    else if (isset($_GET['format']))
      $format = $_GET['format'];
    else
      $format = null;

    $dt_tz = $this->tz();

    $formatter = build_formatter($format);
    if (is_null($formatter))
      $this->renderErrorAndHelp(L("invalid format: ") . $format);
    else {
      if (isset($_GET['save_to_local'])) {
        $file_name = $_GET['save_to_local'];
        if (preg_match('/^[-_a-zA-Z0-9]+$/', $file_name))  {
          $formatter->saveToLocal($file_name);
          $formatter->setAppendRowCount(true);
          if (isset($_GET['zip_threshold'])) {
            $zip_threshold = floatval($_GET['zip_threshold']);
            $formatter->setZipThreshold($zip_threshold);  
          }
        }
        else
          $this->renderErrorAndHelp(L("invalid file name: ") . $file_name . L(" (only alphbet, number, dash and underscore are valid)"));
      }
    }
    
    // handle 'file' parameter
    $download_file_name = null;
    $file_param = array_key_exists('file', $_GET) ? $_GET['file'] : null;
    if (!is_null($file_param)) {
      if (preg_match('/^[-_a-zA-Z0-9]+$/', $file_param))
        $download_file_name = $file_param;
      else
        $this->renderErrorAndHelp(L("invalid file name: ") . $file_param . L(" (only alphbet, number, dash and underscore are valid)"));
    }

    //build and do sql query
    $db = DBConn::instance()->history_db();
    $stmt = $db->prepare($query);
    if (!isset($stmt) || !$stmt) {
      $db_error = $db->errorInfo();
      if (count($db_error) != 3)
        $this->renderErrorAndHelp(L('unknown error'));
      else
        $this->renderErrorAndHelp(L('error: ') . $db_error[2]);
    }
    $stmt->execute();
    
    output_history_data($stmt, $formatter, $download_file_name, null, null, null, null, $dt_tz);
  }
  
  protected function renderErrorAndHelp($error) {
    if ($this->isAjax()) {
      $response = array();
      $this->renderAjaxError($response, $error);
      return;
    }

    $style = <<<STYLEDOC
<style>
.error_text { color: red; }
table, th, td { border: 1px solid grey; }
th,td { padding:15px; }
</style>

STYLEDOC;

    $help_desc = <<<HELPDOC
<div>
    <p>sql_query script will allow you to run a sql query on the history database, the URL syntax looks like this: <b>http://DEVICE_IP/sdcard/cpt/app/sql_query.php?PARAM=VALUE&PARAM2=VALUE2...</b></p>

    <div>Parameter details:
    <table style="border:1px solid grey;">
        <thead>
            <tr>
                <td>Name</td>
                <td>Value</td>
            </tr>
        </thead>
        <tbody>
            <tr>
                <td>sql</td>
                <td>sql query, required.</td>
            </tr>
            <tr>
                <td>format</td>
                <td>returned data format, valid format value: csv, html, json. if missing, the default is html.</td>
            </tr>
            <tr>
                <td>dt_tz</td>
                <td>convert 'dt' column into this timezone. NOTE: the '/' character in the timezone's name must be encoded as '%2F', for example, 'Asia/Shanghai' should be 'Asia%2FShanghai'. </td>
            </tr>
            <tr>
                <td>file</td>
                <td>with this parameter in url value, the returned data will trigger a download file process in browser and the file's name will the content of this parameter; otherwise data will be rendered on browser page directly. for example, if this parameter is 'last_week_data' and 'format' parameter is 'csv', then the download file's default name will be 'last_week_data.csv'. if missing, the default is rendering on browser page.</td>
            </tr>
            <tr>
                <td>save_to_local</td>
                <td>save queried data on controller, you can specify file's name here(for example: "last_month_temperature"). the response will be data file's full path on controller. </td>
            </tr>
            <tr>
                <td>auth_key</td>
                <td>used auth_key if you want to use sql_query under password free mode, ask it from admin of CPT Graphics. </td>
            </tr>
        </tbody>
    </table>
    </div>
</div>
HELPDOC;
    $error_msg = $style . "<div class='error_text'>ERROR: " . $error . "</div>" . $help_desc;
    $this->renderErrorResponse($error_msg);
  }
};

$controller = new SqlQueryController();
$controller->run();

?>
