<?php
//vim: ts=2 sw=2

include_once "db.php";
include_once "redis.php";
include_once "base_controller.php";

function redis_error($error)
{
  throw new error($error);
}

class SigninController extends BaseController {

  private $redis = null;

  public function doGet() {
    if ($this->isSignedin()) {
      $this->redirect($this->makeUrl("app/graphic.php"));
    } else if (!empty($_GET['success_url'])) {
      $_SESSION['url_required'] = $this->sanitizeURL($_GET['success_url']);
    } else if (!empty($_GET['url_required'])) {
      $_SESSION['url_required'] = $this->sanitizeURL($_GET['url_required']);
    }
  }
  
  public function doAjaxGet() {
    $response = array();
    $u = new User();
    $name = $_GET['user']['name'];
    if (!$u->find($name)) {
      $this->renderAjaxError($response, sprintf(L("can not find user %s"), $name));
    }
    $randPart = randStr();
    $_SESSION['authTokenRand'] = $randPart;
    $authToken = $u->attr('salt') . '_' . $randPart;
    $response['authToken'] = $authToken;
    $this->renderAjaxSuccess($response);
  }
  
  protected function doAuth($u) {
    $password = $_POST['user']['password'];
    $authHash = $_POST['user']['authHash'];
    if (isset($password)) {
      return $u->authenticate($password);
    } else if (isset($authHash)) {
      $authRand = $_SESSION['authTokenRand'];
      unset($_SESSION['authRand']);
      return $u->authenticate2($authHash, $authRand);
    } else {
      return false;
    }
  }

  protected function getRedis() {
    if (!file_exists('/usr/bin/redis-server'))
      return null;

    if ($this->redis == null) {
      $this->redis = new redis_cli();
      $this->redis->set_error_function('redis_error');
      $this->redis->connect();
    }
    return $this->redis;
  }

  protected function getFailedAuthCount($name) {
    $redis = $this->getRedis();
    if (!$redis)
      return 0;
    return intval($redis->cmd('HGET', 'web_failed_auth_counter', $name)->get());
  }

  protected function recordFailedAuth($name) {
    $redis = $this->getRedis();
    if (!$redis)
      return;
    $redis->cmd('HINCRBY', 'web_failed_auth_counter', $name, 1)->set();
  }

  protected function resetFailedAuthCount($name) {
    $redis = $this->getRedis();
    if (!$redis)
      return;

    $redis->cmd('HDEL', 'web_failed_auth_counter', $name)->cmd('HDEL', 'web_lockout_timer', $name)->set();
  }

  protected function lockOutAccount($name, $failedAuthCount) {
    if ($failedAuthCount < 3)
      return false;

    $redis = $this->getRedis();
    if (!$redis)
      return false;

    $now = time();
    $startTime = intval($redis->cmd('HGET', 'web_lockout_timer', $name)->get());
    $lockout = false;
    if ($startTime == 0) {
      $lockout = true;
      $redis->cmd('HSET', 'web_lockout_timer', $name, $now)->set();
    }
    else {
      $maxLockoutTime = min(pow(2, $failedAuthCount), 3600); //lock out at most for 1hr
      $lockout = ($now - $startTime <= $maxLockoutTime+rand(1, 5)); //continue to lockout account
    }

    return $lockout;
  }

  public function doAjaxPost() {
    $response = array();
    $u = new User();
    $name = $_POST['user']['name'];
    if (!$u->find($name)) {
      $this->renderAjaxError($response, sprintf(L("can not find user %s"), $name));
    }

    $failedAuthCount = $this->getFailedAuthCount($name);
    if ($failedAuthCount >= 3 && $this->lockOutAccount($name, $failedAuthCount))
    {
      $this->recordFailedAuth($name);
      $this->renderAjaxError($response, L("password is not correct"));
    }

    $remember_me = $_POST['remember_me'];
    if ($this->doAuth($u)) {
      $this->resetFailedAuthCount($name);
      if ($remember_me == 'true')
        $expiry = time() + 2592000; //60*60*24*30
      else
        $expiry = 0;

      setcookie(session_name(), session_id(), $expiry);

      $_SESSION['user_id'] = $u->attr("id");
      $_SESSION['user_name'] = $u->attr("name");
      $_SESSION['last_visit_time'] = time();
      
      $this->signinPhpLiteAdmin($u->attr("checksum"));

      $redirectUrl = $this->userHomePage();
      if (!empty($_SESSION['url_required']))
      {
        $redirectUrl = $this->makeUrl($_SESSION['url_required']);
        unset($_SESSION['url_required']);
      }
      die(<<<EOS
{"redirectUrl": "$redirectUrl" }
EOS
    );
    } else {
      $this->recordFailedAuth($name);
      $this->renderAjaxError($response, L("password is not correct"));
    }
  }
}

$controller = new SigninController();
$controller->run();

?>

<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="">

    <title><?php echo L('Signin') ?> <?php echo Config::$webPageTitle ?></title>

    <!-- Bootstrap core CSS -->
    <link rel="stylesheet" href="../css/bootstrap.min.css" media="screen" />

    <!-- Custom styles for this template -->
    <link href="../css/signin.css" rel="stylesheet">

    <!-- HTML5 shim and Respond.js IE8 support of HTML5 elements and media queries -->
    <!--[if lt IE 9]>
      <script src="../js/html5shiv.js"></script>
      <script src="../js/respond.min.js"></script>
    <![endif]-->
  </head>

  <body>

    <div id="languages" class="row">
        <select>
        <?php 
          $curLang = getLang();
          if (is_null($curLang))
            $curLang = 'en';
          foreach($availableTranslations as $index => $tr) {
            if ($curLang == $tr[0])
                echo "<option value='{$tr[0]}' selected>{$tr[1]}</option>";
            else
                echo "<option value='{$tr[0]}'>{$tr[1]}</option>";
          }
        ?>
        </select>
    </div>

    <div class="container">

      <div class="alert hide">
      </div>

      <div class="row-fluid">
        <div id="logoDiv" class="span6">
          <img src="<?php echo Config::$loginLogoPath ?>" alt="<?php echo Config::$webPageTitle ?>" />
        </div>

        <div id="signinForm" class="span6">
        </div>
      </div>

    </div> <!-- /container -->


    <!-- Backbone templates -->
    <div id='signinFormContent' class="hide">
      <form class="form-signin" method="post" autocomplete="nope">
        <h2 class="form-signin-heading"><?php echo L('Please sign in') ?></h2>
        <input type="text" name="user[name]" id="username" class="input-block-level" placeholder="<?php echo L('User Name') ?>" autofocus>
        <input type="password" name="user[password]" id="password" class="input-block-level" placeholder="<?php echo L('Password') ?>">
        <label class="checkbox">
          <input type="checkbox" name="remember_me" > <?php echo L('Remember me') ?>
        </label>
        <button name="commit" id="signinBtn" class="btn btn-large btn-primary" type="submit"><?php echo L('Sign in') ?></button>
      </form>
    </div>

    <!-- Bootstrap core JavaScript
    ================================================== -->
    <!-- Placed at the end of the document so the pages load faster -->

    <?php frontendLangJson() ?>
    <?php include ("check_browser.php") ?>

    <script type="text/javascript" src="../js/l18n.js"></script>

    <script type="text/javascript" src="../js/json2.js"></script>
    <script type="text/javascript" src="../js/jquery-1.10.0.min.js"></script>
    <script type="text/javascript" src="../js/underscore-min.js"></script>
    <script type="text/javascript" src="../js/underscore.string.min.js"></script>
    <script type="text/javascript" src="../js/backbone-min.js"></script>
    <script type="text/javascript" src="../js/sha1.js"></script>
    <script type="text/javascript" src="../js/signin.js"></script>

  </body>
</html>

